package scj.analyze.statistics.source.counter;

import it.unimi.dsi.fastutil.ints.Int2IntMap;
import it.unimi.dsi.fastutil.ints.Int2IntOpenHashMap;

import java.text.DecimalFormat;
import java.util.Iterator;
import java.util.Map.Entry;

import scj.input.DataTuple;


/*
 * Als Indikator für die Verteilung der Mengenelemente auf die einzelnen Mengen schlage ich vor: 
 * 	"Anteil der Mengenelemente an der Gesamtmenge, welche in mehr als x% der Mengen enthalten sind".
 * Hier zeigt sich, dass die 4 Datensätze sich in diesem Indikator deutlich unterscheiden. 
 * Während bms, flickr und kosarak alle einen niedrigen Anteil an Mengenelementen haben, 
 * welche oft auftauchen, ist dieser bei netflix höher. 
 * 
 * Tabelle:
Dataset: bms-pos
	 0.05 	 0.1 	 0.15 	 0.2 	 0.25 	 0.3 	 0.4 	 0.5 	 0.6 	 0.7 	 0.8 	 0.9 
	 45.99	 27.65	 18.7	 16.05	 9.17	 9.17	 9.17	 9.17	 0	 0	 0	 0
Dataset: flickr
	 0.05 	 0.1 	 0.15 	 0.2 	 0.25 	 0.3 	 0.4 	 0.5 	 0.6 	 0.7 	 0.8 	 0.9 
	 8.83	 6.68	 4.32	 4.32	 4.32	 4.32	 4.32	 0	 0	 0	 0	 0
Dataset: kosarak
	 0.05 	 0.1 	 0.15 	 0.2 	 0.25 	 0.3 	 0.4 	 0.5 	 0.6 	 0.7 	 0.8 	 0.9 
	 25.86	 20.11	 20.11	 17.65	 17.65	 17.65	 13.11	 7.5	 7.5	 0	 0	 0
Dataset: netflix
	 0.05 	 0.1 	 0.15 	 0.2 	 0.25 	 0.3 	 0.4 	 0.5 	 0.6 	 0.7 	 0.8 	 0.9 
	 64.28	 45.83	 33.75	 22.78	 12.8	 6.71	 1.23	 0	 0	 0	 0	 0

 * So zeigt sich bei jedem Messpunkt bis 20%, dass netflix der Anteil der als häufig definierten Mengenelemente am höchsten ist. 
 * Definiert man "häufig" als mehr als 25%, so übersteigt der Anteil bei kosarak den von netflix. Dies legt die Vermutung nahe, 
 * dass hier eine Dreiteilung in "selten", "häufig" und "sehr häufig" für die weitere Auswertung zielführend sein könnte. 
 *  "selten"   x < 0.05
 * 	"häufig"   x= [.05, .2](?)
 *  "sehr häufig"	x = [>= .25](?)
 * (...)
 * 
 * These: 
 *  sehr häufig -> helfen oben nicht beim trennen, schaden unten nicht beim scannen
 * 	viele häufige -> gute klassen-einteilung des baums in 10-100 klassen, wenn man "alle seltenen" als eine klasse sieht. gut für MFU. Sind Klassen nützlich?
 *  
 * 
 * Methodik:
 * - Warum % und nicht Absolut?
 * 
 * Einfluss:
 * Mengenelemente, die in sehr vielen Mengen enthalten sind, werden bei MFU nach vorn sortiert. 
 * Da die zugehörigen Präfixbaumknoten dann sehr sehr viele Einträge in einem Subbaum haben, sind sie 
 * gut geeignet, auf früher Baumebene die Mengen in Klassen aufzuteilen, welche separat voneinander untersucht werden. (...)
 * - riesige II-Einträge
 * - bei LFU: lange Ketten, die eh kaum noch filtern
 * - 
 * 
 */
public class FrequentItemCounterByPercent implements Counter {

	// Property.
	protected double[] percentages = new double[]{.05, .1, .15, .2, .25, .3, .4, .5, .6, .7, .8, .9};
	
	// Simple stats.
	protected int itemCount;
	protected int setCount;
	protected Int2IntMap countMap;
	
	// Aggregated stats.
	protected double avgSetSize;
	protected Int2IntMap frequencyMap;
	protected int frequentItemCount;
	protected double[] frequentItemPercentage;
	
	public FrequentItemCounterByPercent() {
		itemCount = 0;
		setCount = 0;
		countMap = new Int2IntOpenHashMap();
	}
	
	@Override
	public void count(DataTuple tuple) {
		setCount += 1;
		itemCount += tuple.getSet().length;
		
		for(int val: tuple.getSet()) {
			addToCountingMap(countMap, val);
		}
		
	}

	protected void addToCountingMap(Int2IntMap countMap, int val) {
		if(countMap.containsKey(val)) {
			countMap.put(val, 1 + countMap.get(val));
		} else {
			countMap.put(val, 1);
		}
	}

	/* (non-Javadoc)
	 * @see scj.analyze.statistics.source.Counter#aggregate()
	 */
	@Override
	public void aggregate() {
		
		// Build averages
		avgSetSize = ((double) itemCount) / setCount;
		
		// Build frequency map ("histogram")
		initFrequencyMap();
		
		for (Iterator<Entry<Integer, Integer>> iterator = countMap.entrySet().iterator(); 
				iterator.hasNext();) {
			Entry<Integer, Integer> entry = iterator.next();
			addToCountingMap(frequencyMap, entry.getValue());
		}

		int i=0;
		frequentItemPercentage = new double[percentages.length];
		for(double percentage: percentages) {
		
			double threshold = percentage * setCount; //"x% der sets"
			
			frequentItemCount = 0;		
			for (Iterator<Entry<Integer, Integer>> iterator = frequencyMap.entrySet().iterator(); 
					iterator.hasNext();) {
				Entry<Integer, Integer> entry = iterator.next();
				
				if(entry.getKey() > threshold) {
					frequentItemCount += entry.getValue()*entry.getKey();
				}
			}
			
			frequentItemPercentage[i++] = ((double) frequentItemCount) / itemCount;
		}
	}

	protected void initFrequencyMap() {
		frequencyMap = new Int2IntOpenHashMap();
	}

	/* (non-Javadoc)
	 * @see scj.analyze.statistics.source.Counter#output()
	 */
	@Override
	public String output() {
		StringBuilder tmp = new StringBuilder();
		DecimalFormat format = new DecimalFormat("#.##");

		int i=0;
		for(double percentage: percentages) {
			tmp.append("\t "+format.format(percentage)+" ");
		}
		tmp.append("\n");
		
		i=0;
		for(double percentage: percentages) {
			tmp.append("\t "+format.format(frequentItemPercentage[i++]*100)+"");
		}
		
		return tmp.toString();
	}

}
