package scj.runtime;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.github.jamm.MemoryMeter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import scj.algorithm.limitremade.structures.LIMITOPJInvertedIndex;
import scj.algorithm.pretti.PrettiInvertedIndex;
import scj.evaluation.Executor;


public class RuntimeCalculator {
	private static Logger LOGGER;
	
	private Map<String, Long> start;
	private Map<String, Long> end;
	protected Class<?> viewnClass;
    private MemoryMeter meter = new MemoryMeter().withTrackerProvider(MyAlwaysEmptySet.provider());
    private MemoryMeter prettiMeter = new MemoryMeter();

    public RuntimeCalculator() {
		start = new HashMap<String, Long>();
		end = new HashMap<String, Long>();
		LOGGER = LoggerFactory.getLogger(RuntimeCalculator.class);
    }
	
	public RuntimeCalculator(Class<?> clazz) {
		this.viewnClass = clazz;
		start = new HashMap<String, Long>();
		end = new HashMap<String, Long>();
		LOGGER = LoggerFactory.getLogger(clazz);
    }
	
	public String toString() {
		if(!start.isEmpty()) {
			Set<String> keySet = new HashSet<String>();
			keySet.addAll(this.start.keySet());
			keySet.addAll(this.end.keySet());
			String string = "";
			for(String key : keySet) {
				string += getLogPrefix() + key + ":::"+this.getRuntime(key)+"\n";
			}
			return string;
		}
		return "";		
	}

    private String getLogPrefix() {
        return (viewnClass != null ? viewnClass.getName() +": " : "");
    }

    public long getRuntime(String key) {
		if (this.end.containsKey(key) && this.start.containsKey(key) )
			return this.end.get(key) - this.start.get(key);
		else
			return -1;
	}

	public void start(String key) {
		this.start.put(key, System.currentTimeMillis());
	}
	
	public void stop(String key) {
		this.end.put(key, System.currentTimeMillis());
	}

	public <T> T measure(Executable<T> executable, String name) {
		
		T result;
		if(Executor.MODE == Executor.EXECUTION_MODE.Space) {
            result = executable.execute();

            if(result instanceof PrettiInvertedIndex) {
                LOGGER.info("{}Memory usage of {} (kb): {}", getLogPrefix(), name, toKB(prettiMeter.measureDeep(result)));
            } else if(result instanceof LIMITOPJInvertedIndex) {
                LOGGER.info("{}Memory usage of {} (kb): {}", getLogPrefix(), name, toKB(prettiMeter.measureDeep(result)));
            } else {
                LOGGER.info("{}Memory usage of {} (kb): {}", getLogPrefix(), name, toKB(meter.measureDeep(result)));
            }

        } else {
			this.start(name);
			
			result = executable.execute();
			
			this.stop(name);
		}
		
		
		return result;
	}

    private long toKB(long l) {
        return l/1024;
    }


}
