package scj.evaluation;

import java.util.Set;

import scj.algorithm.ConfigurableRawDataAlgorithm;
import scj.algorithm.RawDataAlgorithm;
import scj.algorithm.parallel.adaptiveranging.AdaptiveRanging;
import scj.algorithm.parallel.adaptiverangingqueuing.adaptiveranging.AdaptiveRangingWithPriority;
import scj.algorithm.parallel.listtask.L1ListTask;
import scj.algorithm.parallel.nodetask.OneTask;
import scj.algorithm.parallel.nodetask.switching.SwitchByLevel;
import scj.algorithm.parallel.nodetask.switching.SwitchBySize;
import scj.algorithm.parallel.rangetask.L1RangeTask;
import scj.algorithm.pretti.Pretti;
import scj.algorithm.twotrees.labeling.flat.*;
import scj.algorithm.twotrees.labeling.listrecursion.FlatListBasedRecursion;
import scj.algorithm.twotrees.labeling.minmax.FlatBothMinMax;
import scj.algorithm.twotrees.labeling.minmax.FlatBothMinMaxListBasedRecursion;
import scj.algorithm.twotrees.labeling.sa.LabelingSA;
import scj.input.DataTuple;
import scj.algorithm.order.AscendingOrder;
import scj.algorithm.order.FrequencyOrder;
import scj.algorithm.order.InverseFrequencyOrder;
import scj.algorithm.order.SortOrder;
import scj.result.CountResultList;
import scj.result.Result;
import scj.result.ResultList;
import scj.runtime.RuntimeCalculator;
import scj.algorithm.twotrees.labeling.flat.semi.SemiFlatNone;


public class Executor {
	
	public static enum EXECUTION_MODE {Time, TimeBuildUp, Space};
	public static EXECUTION_MODE MODE = EXECUTION_MODE.Time;
	
	
	/*
	 * Current lgorithms.
	 */
	public static ConfigurableRawDataAlgorithm[] getAllWorkingAlgorithms() {
		ConfigurableRawDataAlgorithm[] algorithms = {
				new Pretti(),
				new LabelingSA(),
				new FlatBoth(),
				new FlatBothForSpaceMeasuring(),
				new FlatLeft(),
				new FlatRight(),
				new FlatNone(),
				new SemiFlatNone(),
				new FlatListBasedRecursion().withSortOrder(new FrequencyOrder()),
				new FlatBothMinMax(),
				new FlatBothMinMaxListBasedRecursion(),
				new OneTask(),
                new SwitchBySize(),
                new SwitchByLevel(),
                new L1ListTask(),
                new L1RangeTask(),
                new AdaptiveRanging(),
                new AdaptiveRangingWithPriority()
		};
		return algorithms;
	}

	public static ConfigurableRawDataAlgorithm[] getAllParallelWorkingAlgorithms() {
		ConfigurableRawDataAlgorithm[] algorithms = {
				new OneTask(),
                new SwitchBySize(),
                new SwitchByLevel(),
                new L1ListTask(),
                new L1RangeTask(),
                new AdaptiveRanging(),
                new AdaptiveRangingWithPriority()
		};
		return algorithms;
	}


	public static ConfigurableRawDataAlgorithm[] getBenchmarkAlgorithms() {

		FrequencyOrder order = new FrequencyOrder();
		
		ConfigurableRawDataAlgorithm[] algorithms = new ConfigurableRawDataAlgorithm[2];
		algorithms[0] = new LabelingSA().withSortOrder(order);
		algorithms[1] = new FlatNone().withSortOrder(order);
		
		return algorithms;
	}
	
	
	
	private RawDataAlgorithm algorithm;

	public Executor(RawDataAlgorithm algorithm) {
		this.algorithm = algorithm;
	}

	public static void main(String[] args) {
		run(CountResultList.class, args[0], MODE, getBenchmarkAlgorithms());
		
	}


    public static void run(Class<? extends Result> resultType, String dataSet, EXECUTION_MODE mode, ConfigurableRawDataAlgorithm[] algorithms) {
        run(resultType, dataSet, mode, algorithms, -1);
    }

    public static void run(Class<? extends Result> resultType, String dataSet, EXECUTION_MODE mode, ConfigurableRawDataAlgorithm[] algorithms, int lines) {
		
		try {
			RuntimeCalculator rc = new RuntimeCalculator();
		
			ResultList resultList = new ResultList();
			
			for(ConfigurableRawDataAlgorithm alg : algorithms) {
				Executor exec = new Executor(alg);
				exec.setExecutionMode(mode);
				
				Result r = resultType.newInstance().withName(alg.getClass().getName());
			
				Set<DataTuple>[] inputData = getInputData(dataSet, lines);
                exec.preexecute(inputData[0], inputData[1], r);
				
				rc.start(alg.toString() +": execute");
				exec.execute(inputData[0], inputData[1], r);
				rc.stop(alg.toString() +": execute");
				
				r.output();
				
				resultList.add(r);		
				System.gc();
			}

			String compareResults = resultList.compareResults();
			if(compareResults != null) {
				System.out.println(compareResults);
			}
			System.out.println(rc);

		} 
		catch (InstantiationException | IllegalAccessException e) {
			e.printStackTrace();
		}
		
	}

	protected static Set<DataTuple>[] getDataTuplesByOffset(String[] args, final int argsOffset) {
		if (args.length == 0 + argsOffset) {
			System.out.println("You have to specify an input file.");
			System.exit(1);
		}
	
		String dataSetName = args[0 + argsOffset];
		int line_limit = -1;
		if (args.length > 1 + argsOffset) {
			line_limit = Integer.valueOf(args[1 + argsOffset]);
		}
	
		return getInputData(dataSetName, line_limit);
	}


	protected static Set<DataTuple>[] getInputData(String dataSetName,
			int line_limit) {
		Set<DataTuple>[] input = SystemSetup.readFile(dataSetName, line_limit);
	
		return input;
	}

	public static ConfigurableRawDataAlgorithm getAlgorithmByName(String algorithmName) {

		ConfigurableRawDataAlgorithm[] algs = Executor.getAllWorkingAlgorithms();
		for (ConfigurableRawDataAlgorithm alg : algs) {
			if (alg.getClass().getSimpleName().equals(algorithmName)) {
				return alg;
			}
		}
		return null;
	}

	public static ConfigurableRawDataAlgorithm getParallelAlgorithmByName(String algorithmName) {

		ConfigurableRawDataAlgorithm[] algs = Executor.getAllParallelWorkingAlgorithms();
		for (ConfigurableRawDataAlgorithm alg : algs) {
			if (alg.getClass().getSimpleName().equals(algorithmName)) {
				return alg;
			}
		}
		return null;
	}


	protected static SortOrder getSortOrderByOffset(String[] args, final int argsOffset) {
		if (args.length == 0 + argsOffset) {
			System.out.println("You have to specify a sort order.");
			System.exit(1);
		}
	
		String sortOrder = args[0 + argsOffset];
	
		SortOrder order = new AscendingOrder();
		if (sortOrder.equals("freq")) {
			order = new FrequencyOrder();
		} else if (sortOrder.equals("inv-freq")) {
			order = new InverseFrequencyOrder();
		}
		
		return order;
	}


	public void preexecute(Set<DataTuple> example, Set<DataTuple> example2, Result result) {
		this.algorithm.preexecute(example, example2, result);
	}

	public void execute(Set<DataTuple> example, Set<DataTuple> example2, Result result) {
		this.algorithm.execute(example, example2, result);
	}
	
	public void setExecutionMode(EXECUTION_MODE mode) {
		MODE = mode;
	}

}
