package scj.algorithm.twotrees.labeling.sa;

import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import scj.algorithm.ConfigurableRawDataAlgorithm;
import scj.algorithm.TreeType;
import scj.algorithm.twotrees.Search;
import scj.algorithm.twotrees.labeling.flat.FlatSearch;
import scj.evaluation.Executor;
import scj.input.DataTuple;
import scj.algorithm.order.InverseFrequencyOrder;
import scj.algorithm.order.SortOrder;
import scj.result.Result;
import scj.runtime.Executable;
import scj.runtime.RuntimeCalculator;
import scj.algorithm.tree.Node;

public class LabelingSA extends ConfigurableRawDataAlgorithm {

	private final Logger LOGGER;

	private SortOrder sortOrder;

	public LabelingSA() {
		sortOrder = new InverseFrequencyOrder();
		LOGGER = LoggerFactory.getLogger(toString());
	}
	
	public void setSortOrder(SortOrder order) {
		sortOrder = order;
	}
	
	public LabelingSA withSortOrder(SortOrder order) {
		this.setSortOrder(order);
		return this;
	}
	
	public String toString() {
		return this.getClass().getName()+"[order="+this.sortOrder+"]";
	}

	@Override
	public void preexecute(Set<DataTuple> set1, Set<DataTuple> set2, Result result) {
		sortOrder.initialize(set2);
		for(DataTuple r : set1) {
			sortOrder.sort(r.getSet());
		}

		for(DataTuple r : set2) {
			sortOrder.sort(r.getSet());
		}
	}

	@Override
	public void execute(final Set<DataTuple> n_a, final Set<DataTuple> n_b, Result result) {
		RuntimeCalculator rc = new RuntimeCalculator(this.getClass());

        Node t1 =
				rc.measure(new Executable<Node>() {
                    public Node execute() {
                        return new EncodedTreeCreator().createInnerTree(n_a, sortOrder);
                    }
                }, "build prefix tree 1");

//		LOGGER.debug("Prefix tree of data set 1: {}", t1);

        final EncodedTreeCreator encodedTreeCreator = new EncodedTreeCreator();

        final Node outerPrefixTree =
				rc.measure(new Executable<Node>() {
                    public Node execute() {
                        return encodedTreeCreator.getPrefixTree(n_b, sortOrder, TreeType.OuterTree);
                    }
                }, "build prefix tree 2 - step: prefix tree");

        Node t2=
				rc.measure(new Executable<Node>() {
                    public Node execute() {
                        return encodedTreeCreator.createOuterTreeFromPrefixTree(outerPrefixTree);
                    }
                }, "build prefix tree 2 - step: PIETree");

		LOGGER.info("{}", rc);

        //Do scj on it
        if(Executor.MODE == Executor.EXECUTION_MODE.Time) {
            Search search = new EncodedSearch(t1, t2, result);
            search.search();
        }
	}

}
