package scj.algorithm.tree.rightside;

import it.unimi.dsi.fastutil.ints.*;
import scj.algorithm.tree.Node;
import scj.algorithm.tree.TreeNodeWithIntervalAnnotation;

import java.util.Collection;

@SuppressWarnings("serial")
public class PIETree implements Node {

	Int2ObjectMap<IntList> nameToPreorder;
	IntList rangeEnds;
	int size;
	IntList tupleIdPositions;
	IntList tupleIds;

    public long recursionTime = 0;
    public long scopeTime = 0;

    public PIETree() {
		this.size = 0;
	}

	public PIETree(Node prefixTree) {
		this();
		readFromTree((TreeNodeWithIntervalAnnotation) prefixTree);
	}

    public PIETree(int tupleCount) {

    }

    @Override
	public boolean equals(Object obj) {
		if(obj instanceof PIETree) {
			PIETree objPieTree = (PIETree) obj;
			return nameToPreorder.equals(objPieTree.nameToPreorder) &&
					rangeEnds.equals(objPieTree.rangeEnds) &&
					size == objPieTree.size &&
					tupleIdPositions.equals(objPieTree.tupleIdPositions) &&
					tupleIds.equals(objPieTree.tupleIds)
					;
		}
		return super.equals(obj);
	}

	@Override
	public String toString() {
		return "PIETree [size="+size+", nameToPreorder="+ nameToPreorder +
				", rangeEnds="+rangeEnds+", tupleIdPositions="+tupleIdPositions+", tupleIds="+tupleIds+"]";
	}

	/*
             * Have to maintain the interface :/
             */
	public Collection<Node> getChildren() {
		return null;
	}

	public Node getChild(int contentElement) {
		return null;
	}

	public boolean equalsName(Node t2) {
		return false;
	}

	public boolean containsTuples() {
		return false;
	}

	public boolean greaterThanOrEqualsName(Node c_b) {
		return false;
	}

	public int compareTo(Node o) {
		return 0;
	}

	public int getName() {
		return 0;
	}

	public Collection<Integer> getTupleIds() {
		return null;
	}

	public void addChild(Node node) {
	}

	public void addContent(int i) {
	}

	/*
	 * Overall size of the tree.
	 */
	public int getSize() {
		return this.size;
	}

	/*
	 * Finding tupleIDs.
	 */
	public IntList getIdsByPosition(int scope_start, int scope_end) {
		int end;
		if (scope_end < this.tupleIdPositions.size()) {
			end = (this.tupleIdPositions.getInt(scope_end));
		} else {
			end = (this.tupleIds.size());
		}
		return this.tupleIds.subList(
				this.tupleIdPositions.getInt(scope_start), end);
	}

	/*
	 * Range lookups.
	 */
	public int getRangeEnd(int scope_start) {
		return this.rangeEnds.getInt(scope_start);
	}

	public IntList findRanges(int name, int scope_start) {
        return limitToRangeBinary(getRangeListByName(name), scope_start,
                this.getRangeEnd(scope_start));
    }

    public IntList findAllRanges(int name, IntList scopes) {
        return limitToRangeBinary(getRangeListByName(name), scopes);
    }


    public IntList getRangeListByName(int name) {
        final IntList integers = nameToPreorder.get(name);
        if (integers != null) {
            return integers;
		}
		return new IntArrayList();
	}

	public IntList limitToRangeBinary(IntList rangeList, int scope_start,
			int scope_end) {

		if (rangeList.size() == 0 || rangeList.getInt(0) > scope_end
				|| rangeList.getInt(rangeList.size() - 1) < scope_start) {
			return new IntArrayList();
		}

		int a = 0;
		int b = rangeList.size();

		int a_2 = -1;
		int b_2 = -1;

		int m;

		while (a < b) {

			m = (a + b) >>> 1;

			if (rangeList.getInt(m) < scope_start) {
				a = m + 1;
			} else {
				if (a_2 == -1 && rangeList.getInt(m) <= scope_end) {
					a_2 = m;
					b_2 = b;
				}

				b = m;
			}
		}

		if (a_2 != -1) {
			while (a_2 < b_2) {

				m = (a_2 + b_2) >>> 1;

				if (rangeList.getInt(m) <= scope_end) {
					a_2 = m + 1;
				} else {

					b_2 = m;
				}
			}
		} else {
			a_2 = a;
			if (rangeList.getInt(a) == scope_end) {
				a_2++;
			}
		}

		return new IntArrayList(rangeList.subList(a, a_2));
	}

	protected void readFromTree(TreeNodeWithIntervalAnnotation prefixTree) {
		decorateWithIntervals(prefixTree, 0);
		this.size = prefixTree.getI2();
		this.tupleIdPositions = new IntArrayList(this.size);
		this.tupleIds = new IntArrayList();

		this.nameToPreorder = new Int2ObjectOpenHashMap<IntList>();
		this.rangeEnds = new IntArrayList(this.size);

		pickIntervalsAndTupleIDs(prefixTree);

	}

	private int decorateWithIntervals(TreeNodeWithIntervalAnnotation prefixTree, int i) {
		prefixTree.setI1(i);
		for (Node child : prefixTree.getChildren()) {
			i = this.decorateWithIntervals((TreeNodeWithIntervalAnnotation) child, i + 1);
		}
		prefixTree.setI2(i);
		return i;
	}

	private void pickIntervalsAndTupleIDs(TreeNodeWithIntervalAnnotation prefixTree) {
		int i1 = prefixTree.getI1();
		this.rangeEnds.add(i1, prefixTree.getI2());

		// Pick interval
		int name = prefixTree.getName();
		putNameToPreorder(name, i1);

		// Pick contents
		this.tupleIdPositions.add(i1, this.tupleIds.size());
		this.tupleIds.addAll(prefixTree.getTupleIds());

		// Recurse preordered -> lists are sorted by nature.
		for (Node child : prefixTree.getChildren()) {
			this.pickIntervalsAndTupleIDs((TreeNodeWithIntervalAnnotation) child);
		}

	}

	protected void putNameToPreorder(int name, int preorder) {
		if (this.nameToPreorder.containsKey(name)) {
			this.nameToPreorder.get(name).add(
					preorder);
		} else {
			IntArrayList list = new IntArrayList();
			list.add(preorder);
			this.nameToPreorder.put(name, list);
		}
	}


	public void init(int tupleCount) {
		int sizeEstimation = tupleCount;
		this.tupleIdPositions = new IntArrayList(sizeEstimation);
		this.tupleIds = new IntArrayList(tupleCount);

		this.nameToPreorder = new Int2ObjectOpenHashMap<IntList>();
		this.rangeEnds = new IntArrayList(sizeEstimation);

		this.size = 0;
	}

	public void addRangeEnd(int nodeId, int rangeEndId) {
		//System.out.println("RangeEnd: "+nodeId+" ->"+rangeEndId);
		if(nodeId > size) {
			size = nodeId;
		}
		while(rangeEnds.size()<nodeId) {
			rangeEnds.add(rangeEnds.size(), 0);
		}
		if(rangeEnds.size() != nodeId) {
			rangeEnds.remove(nodeId);
		}
		rangeEnds.add(nodeId,rangeEndId);
	}

	public void addNameToPreorder(int name, int nodeId) {
		//System.out.println("Name to Preorder: "+name+" ->"+nodeId);
		putNameToPreorder(name, nodeId);

	}

	public void addPreorderToContent(int nodeId, int tupleId) {
		//Caution: Insertion must be in-order!
		//System.out.println("Preorder to tupleIds: " + nodeId + " ->" + tupleId);

		int previousListSize = tupleIdPositions.size();
		int tupleCount = this.tupleIds.size();
		while(tupleIdPositions.size() <nodeId) {
			tupleIdPositions.add(previousListSize, tupleCount);
		}
		if(tupleIdPositions.size() == nodeId) {
			this.tupleIdPositions.add(nodeId, tupleCount);
		}
		this.tupleIds.add(tupleId);

	}


    public IntList limitToRangeBinary(IntList rangeList, IntList scopes) {
   		long tmp = System.currentTimeMillis();
        final IntList recurse = recurse(rangeList, scopes);

        recursionTime += System.currentTimeMillis()-tmp;

        if(recurse == null) {
            return new IntArrayList(0);
        }
        return recurse;

    }

    private IntList recurse(IntList values, IntList scopes) {
        // Sichere Fälle: kein Inhalt.
        if(values.size() == 0) {
            return values;
        }
        if(scopes.size()==0) {
            return null;
        }
        if(values.getInt(0) > getRangeEnd(scopes.getInt(scopes.size()-1))) {
            return null;
        }
        if(values.getInt(values.size()-1) < scopes.getInt(0)) {
            return null;
        }

        // Nur 1 Scope und der enthält alles ...
        if(scopes.size()==1 && scopes.getInt(0) <= values.getInt(0) && getRangeEnd(scopes.getInt(0)) >= values.getInt(values.size()-1)) {
            return values;
        }

        // Nur 1 Wert, also überprüfe ihn direkt.
        if(values.size() == 1) {
            return isValueInAScope(values.getInt(0), scopes) ? values : null;
        }

        int m = (0 + values.size()) >>> 1;
        int needle = values.getInt(m);

        //Find matching scopes
        // Scopes below and above
        // scopeliste zerlegen in vorderen und hinteren teil, mittlerer kann überschneiden
        long tmp = System.currentTimeMillis();

        final int separatorIdx = getSeparator(scopes, needle) - 1;
        final IntList scopesBelow = scopes.subList(0, separatorIdx + 1);

        IntList scopesAbove;
        if (separatorIdx >= 0 && getRangeEnd(scopes.getInt(separatorIdx)) >= needle) {
            scopesAbove = scopes.subList(separatorIdx, scopes.size());
        } else {
            scopesAbove = scopes.subList(separatorIdx + 1, scopes.size());
        }

        scopeTime += System.currentTimeMillis() - tmp;

        IntList result = recurse(values.subList(0, m), scopesBelow);
        IntList x = recurse(values.subList(m, values.size()), scopesAbove);

        if (result == null && x != null) {
            result = x;
        } else if (result != null & x != null) {
            result = new IntArrayList(result);
            result.addAll(x);
        }
        return result;

    }

    private boolean isValueInAScope(int value, IntList scopes) {

        int l = 0;
        int r = scopes.size();
        int m = 0;

        while(l < r) {
            m = (r + l) / 2;

            final int scopeStartM = scopes.getInt(m);
            if (value >= scopeStartM && value <= getRangeEnd(scopeStartM)) {
                return true;
            }
            if (value > scopeStartM) {
                l = m + 1;
            } else if (value < scopeStartM) {
                r = m - 1;
            }

        }
        final int scopeStartM = scopes.getInt(l);
        if (value >= scopeStartM && value <= getRangeEnd(scopeStartM)) {
            return true;
        }

        return false;
    }

    public int getSeparator(IntList scopes, int needle) {
        //Binäre Suche nach Nadel in der scope_start-Liste
        //Wenn ein scope vor needle startet und danach endet, tue ihn mit rein :)
        // -> Alle scopes mit scope_start <= needle

        final int scopesSize = scopes.size();

        int l = 0;
        int r = scopesSize;
        int m;

        while(l < r) {
            m = (r + l) / 2;

            final int intAtM = scopes.getInt(m);
            if (needle == intAtM) {
                return m;
            }
            if (needle > intAtM) {
                if (m+1 < scopesSize && scopes.getInt(m + 1) > needle) {
                    return m+1;
                }
                l = m + 1;
            } else if (needle < intAtM) {
                r = m - 1;
            }

        }
        if(r+1 < scopesSize) {
            return r + 1;
        }
        return r;
    }
}
