package scj.algorithm.pretti;

import gnu.trove.iterator.TIntIterator;
import gnu.trove.list.array.TIntArrayList;
import it.unimi.dsi.fastutil.ints.IntListIterator;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import scj.algorithm.ConfigurableRawDataAlgorithm;
import scj.algorithm.RawDataAlgorithm;
import scj.input.DataTuple;
import scj.algorithm.order.FrequencyOrder;
import scj.algorithm.order.SortOrder;
import scj.result.Result;
import scj.runtime.Executable;
import scj.runtime.RuntimeCalculator;
import scj.algorithm.tree.Node;
import scj.algorithm.tree.TreeNode;

import java.util.Set;

public class Pretti extends ConfigurableRawDataAlgorithm implements RawDataAlgorithm {
	private static final Logger LOGGER = LoggerFactory.getLogger(Pretti.class);

	SortOrder sortOrder;

	public Pretti(){
		// "every record r in R is internally ordered such that the most frequent item appears first"
		sortOrder = new FrequencyOrder();
	}

	public void preexecute(Set<DataTuple> set1, Set<DataTuple> set2, Result result) {
		sortOrder.initialize(set1);
		for(DataTuple r : set1) {
			r.setSet(sortOrder.sort(r.getSet()));
		}
	}

	
	@Override
	public void execute(final Set<DataTuple> set1, final Set<DataTuple> set2, Result result) {

		RuntimeCalculator rc = new RuntimeCalculator(this.getClass());
		
		PrettiPrefixTree tr = 
			(PrettiPrefixTree) rc.measure(new Executable() {
				public Object execute() {
					return new PrettiPrefixTree(set1);
				}
			}, "build prefix tree");
		System.out.println("prefixtree built");
		//System.out.println("tree size: "+tr.getSize());

		PrettiInvertedIndex is = 
				(PrettiInvertedIndex) rc.measure(new Executable() {
					public Object execute() {
						return new PrettiInvertedIndex(set2);
					}
				}, "build inv index");
		System.out.println("inverted index built");
		//System.out.println("index size: "+is.getSize());

		TIntArrayList cl = 
				(TIntArrayList) rc.measure(new Executable() {
					public Object execute() {
						return TupleIDList.getCandicateListContainingAllEntries(set2);
					}
				}, "get full list");
		
		
		//Pretti join
		for(Node c : tr.getChildren()) {
			this.processNode((PrettiPrefixTreeNode) c, cl, is , result);
		}
		
		System.out.println("DONE");

		LOGGER.info("{}", rc);
		
	}
		
	public void processNode(TreeNode currentNode, TIntArrayList tupleIDCandidates, PrettiInvertedIndex invertedIndex, Result result) {

		//LOGGER.debug("Processing node {} with candidate list {}", currentNode, tupleIDCandidates);

		TIntArrayList tupleIDs = TupleIDList.intersect(
			tupleIDCandidates,
			invertedIndex.get(currentNode.getName())
		);
		

		 // This is not done by PRETTI itself, but made it 10% faster
		if(tupleIDs.size() == 0) {
			return;
		}

		TIntIterator it = tupleIDs.iterator();
		while(it.hasNext()) {
			int s = it.next();
			IntListIterator itr = currentNode.getTupleIds().iterator();
			while (itr.hasNext()){
				Integer r = itr.next();
				result.add(r,s);
			}
		}
		
		for(Node c : currentNode.getChildren()) {
			this.processNode((TreeNode) c, tupleIDs , invertedIndex , result);
		}
	}

    public void setSortOrder(SortOrder sortOrder) {
        this.sortOrder = sortOrder;
    }
}
