package scj.algorithm.limitremade.structures;

import gnu.trove.iterator.TIntIterator;
import gnu.trove.list.array.TIntArrayList;

import java.util.Set;

import scj.input.DataTuple;

/**
 * Auxiliary class for performing intersections.
 * Two intersect methods are implemented, one using binary search
 * and the other one imitating merge sort.
 * A cost model is introduced in judgeISCost() for deciding between
 * these two methods.
 * @author Christopher Schiefer
 *
 */
public class TupleIDList {
	
	private static final int MERGE = 1;
	private static final int BINARY = 2;
	
	private static final double ESTIMATED_SIZE_REDUCTION_AT_INTERSECTION = 0.5;
	// average size of a record in the candidate list, needed for statistics
	private static int averageCandSize;
	
	public static TIntArrayList getCandicateListContainingAllEntries(Set<DataTuple> set2) {
		averageCandSize = 0;
		long candSizeSum = 0;
		TIntArrayList cl = new TIntArrayList(set2.size());
		for(DataTuple tuple: set2) {
			cl.add(tuple.getId());
			candSizeSum += tuple.getSet().length;
		}
		averageCandSize = (int) Math.ceil(candSizeSum / (double)cl.size());
		cl.sort();
		return cl;
	}
	
	public static int getAvgCandSize() {
		return averageCandSize;
	}
	
	/**
	 * Intersect tupleIDCandidates and tIntArrayList either with merge oder binary
	 * method. First use a cost model to decide between these two.
	 */
	public static TIntArrayList intersectHybrid(TIntArrayList tupleIDCandidates, TIntArrayList tIntArrayList, int[][] sArray) {
		int decision = judgeISCost(tupleIDCandidates, tIntArrayList);
//		intersectCnt++;
		if(decision == MERGE) {
			TIntArrayList r = intersectMerge(tupleIDCandidates, tIntArrayList, sArray);
			return r;
		} else if(decision == BINARY) {
			TIntArrayList r = intersectBinary(tupleIDCandidates, tIntArrayList, sArray);
			return r;
		}
		// no other case possible...
		return null;
	}
	
	/**
	 * Decide between binary and merge method, based on the lists' sizes.
	 */
	public static int judgeISCost(TIntArrayList tupleIDCandidates, TIntArrayList tIntArrayList) {
		// decide between binary and merge
		int x = tupleIDCandidates.size();
		int y = tIntArrayList.size();
		// x needs to be the smaller list, so swap if needed
		if(x > y) {
			int tmp = x;
			x = y;
			y = tmp;
		}
		// formulas from Panos' physical cost model
		double binaryISCost = 161*x*Math.log(y)/Math.log(2.0) + 59074;
		double mergeISCost = 292*x + 85*y + 175636;
		// return intersection strategy with lower cost
		if(binaryISCost < mergeISCost) {
			return BINARY;
		} else {
			return MERGE;
		}
	}
	
	/**
	 * Perform binary intersection.
	 */
	public static TIntArrayList intersectBinary(TIntArrayList tupleIDCandidates, TIntArrayList tIntArrayList, int[][] sArray) {
		averageCandSize = 0;
		long candSizeSum = 0;
		// check for empty lists
		if(tupleIDCandidates.size()==0)
			return tupleIDCandidates;
		if(tIntArrayList.size()==0)
			return tIntArrayList;
		
		// create result list with estimated size
		int estimatedListSize = (int) (ESTIMATED_SIZE_REDUCTION_AT_INTERSECTION * Math.min(tupleIDCandidates.size(), tIntArrayList.size()));
		TIntArrayList newList = new TIntArrayList(estimatedListSize);
		
		boolean candLonger = tupleIDCandidates.size() > tIntArrayList.size();
		// iterate through all elements in shorter list and try to find
		// them in the longer list with binary search
		if(candLonger) {
			TIntIterator iterB = tIntArrayList.iterator();
			while(iterB.hasNext()) {
				int b = iterB.next();
				if(binarySearch(tupleIDCandidates, b)) {
					newList.add(b);
					candSizeSum += sArray[b].length;
				}
			}
		} else {
			TIntIterator iterA = tupleIDCandidates.iterator();
			while(iterA.hasNext()) {
				int a = iterA.next();
				if(binarySearch(tIntArrayList, a)) {
					newList.add(a);
					candSizeSum += sArray[a].length;
				}
			}
		}
		
		// used for LIMIT PLUS only
		averageCandSize = (int)Math.ceil(candSizeSum / (double)newList.size());
		
		return newList;
	}
	
	/**
	 * Search for element in tList with binary search.
	 */
	public static boolean binarySearch(TIntArrayList tList, int element) {
		// look for element in tList, return true iff found
		int low = 0;
		int high = tList.size()-1;
		while(low <= high) {
			// element is between tList[low] and tList[high] or not in list
			int mid = low + (high - low) / 2;
			int midElement = tList.get(mid);
			if(element < midElement) {
				high = mid - 1;
			} else if(element > midElement) {
				low = mid + 1;
			} else {
				// found element
				return true;
			}
		}
		return false;
	}
	
	/**
	 * Perform merge intersection.
	 */
	public static TIntArrayList intersectMerge(TIntArrayList tupleIDCandidates, TIntArrayList tIntArrayList, int[][] sArray) {
		averageCandSize = 0;
		long candSizeSum = 0;

		// check for empty lists
		if(tupleIDCandidates.size()==0)
			return tupleIDCandidates;
		if(tIntArrayList.size()==0)
			return tIntArrayList;
		
		// create result list with estimated size
		int estimatedListSize = (int) (ESTIMATED_SIZE_REDUCTION_AT_INTERSECTION * Math.min(tupleIDCandidates.size(), tIntArrayList.size()));
		TIntArrayList newList = new TIntArrayList(estimatedListSize);
		
		TIntIterator iterA = tupleIDCandidates.iterator();
		TIntIterator iterB = tIntArrayList.iterator();
		int b = iterB.next();
		while(iterA.hasNext()) {
			int a = iterA.next();
			while(iterB.hasNext() && (a > b)) {
				b = iterB.next();
			}
			if(!iterB.hasNext() && (a > b)) {
				break;
			}
			if(a == b ) {
				candSizeSum += sArray[a].length;
				newList.add(a);
			} 
		}
		// used for LIMIT PLUS only
		averageCandSize = (int)Math.ceil(candSizeSum / (double)newList.size());
		
		return newList;
	}
}
