package scj.algorithm.limitremade.structures;

import gnu.trove.list.array.TIntArrayList;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;

import java.util.Map;
import java.util.Set;

import scj.input.DataTuple;

/**
 * Index for storing the S relation with LIMIT. 
 * Stores all tuples and returns their IDs when asked for tuples
 * that include a certain item.
 * Only slightly different to the index used by PRETTI,
 * e.g. it works with Arrays as well as Sets.
 * @author Christopher Schiefer
 *
 */
public class LIMITOPJInvertedIndex {
	
	private static final TIntArrayList EMPTY_LIST = new TIntArrayList();
	// map for the actual index
	protected Map<Integer, TIntArrayList> index;
	// count how many tuples are in index
	protected long tupleCount;
	
	public LIMITOPJInvertedIndex() {
		this.index = new Int2ObjectOpenHashMap<TIntArrayList>();
		tupleCount = 0;
	}

	/**
	 * Create new index and insert all tuples from set2.
	 */
	public LIMITOPJInvertedIndex(Set<DataTuple> set2) {
		this.index = new Int2ObjectOpenHashMap<TIntArrayList>();
		tupleCount = 0;
		this.fillIndex(set2);
	}


	/**
	 * Create new index and insert all tuples from set2.
	 */
	public LIMITOPJInvertedIndex(int[][] set2) {
		this.index = new Int2ObjectOpenHashMap<TIntArrayList>();
		tupleCount = 0;
		this.fillIndex(set2);
	}

	/**
	 * Add all tuples in set2 to index
	 */
	public void fillIndex(Set<DataTuple> set2) {
		tupleCount += set2.size();
		for(DataTuple tuple: set2) {
			for(int name: tuple.getSet()) {
				if(!this.index.containsKey(name)) {
					this.index.put(name, new TIntArrayList());
				}
				this.index.get(name).add(tuple.getId());					
				
			}
			
		}	
		for(TIntArrayList tupleIdList: this.index.values()) {
			tupleIdList.sort();
		}
	}

	/**
	 * Add all tuples in set2 to index
	 */
	public void fillIndex(int[][] set2) {
		fillIndex(set2, 1, set2.length);
	}

	/**
	 * Add tuples from set2 in the specified range.
	 */
	public void fillIndex(int[][] set2, int begin, int end) {
		int tCnt = end-begin;
		int name;
		tupleCount += tCnt;
		// iterate through selected tuples
		for(int i=begin; i<end; i++) {
			// iterate through this tuple
			for(int j=0; j<set2[i].length; j++) {
				name = set2[i][j];
				if(!this.index.containsKey(name)) {
					this.index.put(name, new TIntArrayList());
				}
				// position in array signals the ID
				this.index.get(name).add(i);
			}
		}
	}
	
	/**
	 * Return how many keys the index contains.
	 */
	public int getSize() {
		return index.keySet().size();
	}
	
	/**
	 * Return how many records the index contains.
	 */
	public long getTupleCount() {
		return tupleCount;
	}

	/**
	 * Returns all tuples that include the given item.
	 */
	public TIntArrayList get(int i) {
		if(this.index.containsKey(i)) {
			return this.index.get(i);					
		} 
		return EMPTY_LIST;
	}
	
	public String toString() {
		return "Inverted index: "+this.index.toString();
		
	}

}
