package scj.algorithm.limitremade.structures;

import it.unimi.dsi.fastutil.ints.IntArrayList;
import it.unimi.dsi.fastutil.ints.IntList;
import scj.algorithm.tree.Node;
import scj.algorithm.tree.TreeNode;

/**
 * TreeNode implementation used by LIMIT. Includes an additional list
 * for storing 'cut off tuples', i.e. tuples that are longer than the
 * limit for this tree and that end in this node.
 * Also includes a counter for how many records are in this node
 * and all nodes contained by the subtree with this node as its root.
 * @author Christopher Schiefer
 *
 */
public class LIMITAPrefixTreeNode extends TreeNode {

	private static final long serialVersionUID = -7050432969140113983L;

	// stores tuples that are longer than the tree limit
	protected IntList cutOffTuples;
	// count how many records this node and all the nodes 
	// in its subtree below contain
	protected int recsInSubtreeCnt;
	// depth in tree
	protected int depth;

	public LIMITAPrefixTreeNode(int name, int depth) {
		super(name);
		cutOffTuples = new IntArrayList();
		recsInSubtreeCnt = 0;
		this.depth = depth;
	}
	
	public void increaseRecCnt() {
		recsInSubtreeCnt++;
	}
	
	public int getDepth() {
		return depth;
	}
	
	public void printChildren() {
		System.out.println("this: "+getName());
		for(Node c : getChildren()) 
			System.out.println("\t"+c.getName());
	}
	
	public int getRecsInSubtreeCnt() {
		return recsInSubtreeCnt;
	}
	
	public int getSize() {
		int size = 0;
		for(Node child : getChildren()) {
			size++;
			size += ((LIMITAPrefixTreeNode)child).getSize();
		}
		return size;
	}
	
	public int getCutOffCnt() {
		int cnt = cutOffTuples.size();
		for(Node child : getChildren()) {
			cnt += ((LIMITAPrefixTreeNode)child).getCutOffCnt();
		}
		return cnt;
	}
	
	public int getRecordCnt() {
		int cnt = tupleIds.size();
		for(Node child : getChildren()) {
			cnt += ((LIMITAPrefixTreeNode)child).getRecordCnt();
		}
		return cnt;
	}

	
	public IntList getCutOffTuples() {
		return this.cutOffTuples;
	}
	
	public void addCutOffTuple(int tup) {
		this.cutOffTuples.add(tup);
	}

	@Override
	public boolean containsTuples() {
		return !this.tupleIds.isEmpty();
	}
	
	public boolean containsCutOffs() {
		return !this.cutOffTuples.isEmpty();
	}

}
