package scj.algorithm.limitcomplete;

import gnu.trove.iterator.TIntIterator;
import gnu.trove.list.array.TIntArrayList;
import it.unimi.dsi.fastutil.ints.IntListIterator;

import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import scj.algorithm.ConfigurableRawDataAlgorithm;
import scj.algorithm.RawDataAlgorithm;
import scj.algorithm.limitremade.structures.LIMITAPrefixTree;
import scj.algorithm.limitremade.structures.LIMITAPrefixTreeNode;
import scj.algorithm.limitremade.structures.LIMITOPJInvertedIndex;
import scj.algorithm.limitremade.structures.TupleIDList;
import scj.algorithm.order.FrequencyOrder;
import scj.algorithm.order.InverseFrequencyOrder;
import scj.algorithm.order.SortOrder;
import scj.algorithm.tree.Node;
import scj.input.DataTuple;
import scj.result.Result;
import scj.runtime.Executable;
import scj.runtime.RuntimeCalculator;

/**
 * Class is only used for measuring the space needed for LIMITOPJ,
 * since the index is built step by step and difficult to measure otherwise.
 * It is only used for measuring space consumption and does not even perform
 * the join method properly.
 * @author Christopher Schiefer
 *
 */
public class LIMITOPJINDEXING extends ConfigurableRawDataAlgorithm implements RawDataAlgorithm  {
	private static final Logger LOGGER = LoggerFactory.getLogger(LIMITOPJINDEXING.class);

	SortOrder sortOrder;
	static final int SORT_ASC = 0;
	static final int SORT_DESC = 1;
	int sOrder;
//	static int debug = 0;
	static int limit = 4;
//	static boolean DEBUG = false;
//	static boolean SHOWTREE = false;
//	static long candidateCnt;
	int[][] rdArray;
	int[][] sdArray;
	long myresult;

	public LIMITOPJINDEXING(int limit, String so){
		// set limit
		LIMITOPJINDEXING.limit = limit;
		// set sort order
		if(so.equals("ASC")) {
			sOrder = SORT_ASC;
			sortOrder = new InverseFrequencyOrder();
		} else if(so.equals("DESC")) {
			sOrder = SORT_DESC;
			sortOrder = new FrequencyOrder();
		}
		myresult = 0;
//		candidateCnt = 0;
	}

	public void preexecute(Set<DataTuple> set1, Set<DataTuple> set2, Result result) {
		sortOrder.initialize(set1);
		for(DataTuple r : set1) {
			r.setSet(sortOrder.sort(r.getSet()));
		}
		// set2 is also ordered for LIMIT
		sortOrder.initialize(set2);
		for(DataTuple r : set2) {
			r.setSet(sortOrder.sort(r.getSet()));
		}
		
		rdArray = new int[set1.size()+1][];
						for(DataTuple dt : set1) {
							int[] dtSet = dt.getSet();
							rdArray[dt.getId()] = dtSet;
						}
						
		sdArray = new int[set2.size()+1][];
						for(DataTuple dt : set2) {
							int[] dtSet = dt.getSet();
							sdArray[dt.getId()] = dtSet;
						}
	}
	
	// sort dataArray by only rearranging indices pointing on the array.
	// as a result, we return an array with the pointers pointing to
	// the correct order of the dataArray to be read
	public int[] sortRelation(int[][] dataArray) {
		int[] oldIndices = new int[dataArray.length];
		for(int i=1; i<oldIndices.length; i++)
			oldIndices[i] = i;
		
		sortRelationRec(dataArray, oldIndices, 1, dataArray.length-1);
		
		return oldIndices;
	}
	
	public void sortRelationRec(int[][] dataArray, int[] oldIndices, int left, int right) {
		//System.out.println(left+" to "+right+":");
		if(left >= right)
			return; // fewer than 2 elements
		
		swap(dataArray, oldIndices, left, (left+right)/2);
		int last = left;
		int diff, ldiff;
		for(int i=left+1; i <= right; i++) {
			if(sOrder == SORT_ASC) 
				diff = dataArray[i][0] - dataArray[left][0];
			else
				diff = dataArray[left][0] - dataArray[i][0];
			ldiff = dataArray[i].length - dataArray[left].length;
			if((diff > 0) || (diff==0) && ((ldiff < 0) || (ldiff == 0) && (oldIndices[i] < oldIndices[left]))) {
				last = last+1;
				swap(dataArray, oldIndices, last, i);
			}
		}
		swap(dataArray, oldIndices, left, last);
		sortRelationRec(dataArray, oldIndices, left, last-1);
		sortRelationRec(dataArray, oldIndices, last+1, right);
	}
	
	// swaps both elements in the data as well as in the indices array
	private void swap(int[][] dataArray, int[] oldIndices, int left, int right) {
		int oldLeft = oldIndices[left];
		int[] tData = dataArray[left];
		
		dataArray[left] = dataArray[right];
		oldIndices[left] = oldIndices[right];
		
		dataArray[right] = tData;
		oldIndices[right] = oldLeft;
	}

	
	@Override
	public void execute(final Set<DataTuple> set1, final Set<DataTuple> set2, Result result) {
//		System.out.println("LIMIT+(OPJ) with limit="+limit);

		RuntimeCalculator rc = new RuntimeCalculator(this.getClass());
		
//		Map<Integer, int[]> dataMap = new Int2ObjectOpenHashMap<int[]>(set1.size());
//		for(DataTuple dt : set1) {
//			dataMap.put(dt.getId(), dt.getSet());
//		}
		
//		TupleIDList.intersectCnt = 0;
		
		final int[][] rArray = rdArray;
		final int[][] sArray = sdArray;
		rc.measure(new Executable() {
			public Object execute() {
				return rdArray;
			}
		}, "R array");
		rc.measure(new Executable() {
			public Object execute() {
				return sdArray;
			}
		}, "S array");
		
		LIMITAPrefixTree tr = 
			(LIMITAPrefixTree) rc.measure(new Executable() {
				public Object execute() {
					return new LIMITAPrefixTree(rArray, limit);
				}
			}, "build prefix tree2");
//		System.out.println("prefixtree built");
		
		
		int[] oldIndices = 
				(int[]) rc.measure(new Executable() {
					public Object execute() {
						return sortRelation(sArray);
					}
				}, "sort S");
//		System.out.println("S sorted");
		
		// pointer to begin of partition
		int partBegin = 1;
		int sid = 1;
		int pbIndex = 1;

		int[] partBs;
		if(sOrder == SORT_ASC) {
			partBs = new int[sArray[1][0]+1];
		} else {
			partBs = new int[sArray[sArray.length-1][0]+1];
		}
//		System.out.println("partBs.length="+partBs.length);
		partBs[0] = 1;
		
		// determine max value in domain:
		int maxVal;
		if(sOrder == SORT_ASC) {
			maxVal = sArray[1][0];
		} else {
			maxVal = sArray[sArray.length-1][0];
		}
		
		for(int i=1; i<rArray.length; i++) {
			if(rArray[i][0] > maxVal)
				maxVal = rArray[i][0];
		}
		
		if(sOrder == SORT_ASC) {
			for(int kid=maxVal; kid>=0; kid--) {
				// increase sid as long as first element of tuple is kid
				while((sid < sArray.length) && (sArray[sid][0] == kid)) {
					// we don't need to swap IDs here as in Panos' code
					//
					sid++;
				}
				if(partBegin != sid) { // partition's size is 0 otherwise: do nothing
					// fill index with found partition
//					is.fillIndex(sArray, partBegin, sid);
					partBegin = sid;
					partBs[pbIndex] = sid;
					pbIndex++;
					
//					LIMITAPrefixTreeNode c = (LIMITAPrefixTreeNode)tr.getChild(kid);
//					if(c != null) { // are there tuples with kid as first element in R?
//	//					if(DEBUG)
//	//						System.out.println("processing root child "+c.getName());
//	//					System.out.println(c.getName()+"::"+avgCandLength);
//						this.processNode((LIMITAPrefixTreeNode) c, is , result, limit, rArray, sArray, oldIndices);
//					}
				}
			}
		} else if(sOrder == SORT_DESC) {
			for(int kid=sArray[1][0]; kid<=maxVal; kid++) {
				// increase sid as long as first element of tuple is kid
				while((sid < sArray.length) && (sArray[sid][0] == kid)) {
					// we don't need to swap IDs here as in Panos' code
					//
					sid++;
				}
				if(partBegin != sid) { // partition's size is 0 otherwise: do nothing
					// fill index with found partition
//					is.fillIndex(sArray, partBegin, sid);
					partBegin = sid;
					partBs[pbIndex] = sid;
					pbIndex++;
				}
					
//				LIMITAPrefixTreeNode c = (LIMITAPrefixTreeNode)tr.getChild(kid);
//				if(c != null) { // are there tuples with kid as first element in R?
//	//					if(DEBUG)
//	//						System.out.println("processing root child "+c.getName());
//	//					System.out.println(c.getName()+"::"+avgCandLength);
//					this.processNode((LIMITAPrefixTreeNode) c, is , result, limit, rArray, sArray, oldIndices);
//				}
			}
		}
		
		final int[] bounds = partBs;

		// measure index
		LIMITOPJInvertedIndex is = 
				(LIMITOPJInvertedIndex) rc.measure(new Executable() {
					public Object execute() {
						LIMITOPJInvertedIndex ind = new LIMITOPJInvertedIndex();
						for(int i=1; i<bounds.length; i++) {
							ind.fillIndex(sArray, bounds[i-1], bounds[i]);
						}
						return ind;
					}
				}, "build inv index");
//		System.out.println("index size="+is.getTupleCount());
		
		
		
//		System.out.println("items="+(maxVal+1));
//		System.out.println("tuples="+(sArray.length-1));
//		int maxSetSize = 0;
//		long setSizeSum = 0;
//		for(int i=1; i<sArray.length; i++) {
//			if(sArray[i].length > maxSetSize)
//				maxSetSize = sArray[i].length;
//			setSizeSum += sArray[i].length;
//		}
//		System.out.println("maxSetSize="+maxSetSize);
//		double avgSetSize = (setSizeSum) / (double)(sArray.length-1);
//		System.out.println("avgSetSize="+avgSetSize);
		
//		System.out.println("DONE");
//		System.out.println("Candidate count: "+candidateCnt);
//		System.out.println("Intersection count: "+TupleIDList.intersectCnt);

		LOGGER.info("{}", rc);
	}
	
	public void processNode(LIMITAPrefixTreeNode currentNode, LIMITOPJInvertedIndex invertedIndex, Result result, int limit, int[][] rArray, int[][] sArray, int[] oldSIndices) {
	//	if(SHOWTREE)
	//		System.out.println("node: "+fatherNode+"->"+currentNode.getName());
	//	fatherNode += "->"+currentNode.getName();
		//LOGGER.debug("Processing node {} with candidate list {}", currentNode, tupleIDCandidates);
	
		TIntArrayList tuples = invertedIndex.get(currentNode.getName());
//		int avgCandLength = 0;
//		int candLengthSum = 0;
//		TIntIterator tupitr = tuples.iterator();
//		while(tupitr.hasNext()) {
//			candLengthSum += sArray[tupitr.next()].length;
//		}
//		avgCandLength = (int)Math.ceil(candLengthSum / (double)tuples.size());
	//	System.out.println(currentNode.getName()+": avgCand="+candLengthSum+"/"+tuples.size()+"="+avgCandLength);
		
	//	if(DEBUG)
	//		System.out.println("average cand length: "+avgCandLength);
	//		int newAvgCandLength = TupleIDList.getAvgCandSize();
		

		 // This is not done by PRETTI itself, but made it 10% faster
		if(tuples.size() == 0) {
			return;
		}

		verifySmart(currentNode, tuples, limit, rArray, sArray, oldSIndices, result);
		
		for(Node c : currentNode.getChildren()) {
			this.processNodeRecursion((LIMITAPrefixTreeNode) c, tuples , invertedIndex , result, limit, rArray, sArray, oldSIndices);
		}
	}
		
	public void processNodeRecursion(LIMITAPrefixTreeNode currentNode, TIntArrayList tupleIDCandidates, LIMITOPJInvertedIndex invertedIndex, Result result, int limit, int[][] rArray, int[][] sArray, int[] oldSIndices) {
//		if(SHOWTREE)
//			System.out.print("node: "+fatherNode+"->"+currentNode.getName());
//		fatherNode += "->"+currentNode.getName();
		//LOGGER.debug("Processing node {} with candidate list {}", currentNode, tupleIDCandidates);

//		if(continueAsLIMIT(currentNode, invertedIndex, tupleIDCandidates, avgCandLength)) {
//			if(SHOWTREE)
//				System.out.println();
			// Strategy A, proceeding normally as LIMIT
			TIntArrayList tuples = TupleIDList.intersectHybrid(
				tupleIDCandidates,
				invertedIndex.get(currentNode.getName()),
				sArray
			);
//			int newAvgCandLength = TupleIDList.getAvgCandSize();
//			if(DEBUG)
//				System.out.println("new average cand length: "+newAvgCandLength);
			
			 // This is not done by PRETTI itself, but made it 10% faster
			if(tuples.size() == 0) {
				return;
			}
			
			verifySmart(currentNode, tuples, limit, rArray, sArray, oldSIndices, result);
			
			for(Node c : currentNode.getChildren()) {
				this.processNodeRecursion((LIMITAPrefixTreeNode) c, tuples , invertedIndex , result, limit, rArray, sArray, oldSIndices);
			}
//		} else {
////			if(SHOWTREE)
////				System.out.println("[c]");
//			// Follow strategy B, not continuing as LIMIT	
//			// depth-1 many elements are already checked by previous traversal of the tree
//			int safe = currentNode.getDepth()-1;
//			verifyNode(currentNode, tupleIDCandidates, safe, rArray, sArray, oldSIndices, result);
//			
//			for(Node child : currentNode.getChildren())
//				recCheckNode((LIMITAPrefixTreeNode)child, tupleIDCandidates, safe, rArray, sArray, oldSIndices, result);
//		}
	}
	
	public void verifySmart(LIMITAPrefixTreeNode currentNode, TIntArrayList tupleIDCandidates, int safe, int[][] rArray, int[][] sArray, int[] oldSIndices, Result result) {
		TIntIterator it = tupleIDCandidates.iterator();
		//System.out.println("verifyNode: "+currentNode.getName());
	
		// run through candidate list
		while(it.hasNext()) {
			int s = it.next();
			// are there tuples in this node for verification?
			if(currentNode.containsTuples()) {
				IntListIterator itr = currentNode.getTupleIds().iterator();
				while (itr.hasNext()){
					int r = itr.next();
//					candidateCnt++;
					result.add(r, oldSIndices[s]);
					myresult++;
				}
			}
			// are there cut off tuples to verify?
			if(currentNode.containsCutOffs()) {
				IntListIterator itr = currentNode.getCutOffTuples().iterator();
				int[] sSet = sArray[s];
				while (itr.hasNext()){
					int r = itr.next();
					int[] rSet = rArray[r];
					if(qualify(rSet, sSet, safe)) {
						result.add(r, oldSIndices[s]);
						myresult++;
					}
				}
			}
		}
	}
	
	
	public boolean qualify(int[] rSet, int[] sSet, int limit) {
		// uses sort order because either the first or last few ('limit') items
		// of each tuple are checked already
//		candidateCnt++;
		
		if(sOrder == SORT_DESC) {
			int sCnt = limit;
			for(int rCnt = limit; rCnt < rSet.length; rCnt++) {
				int element = rSet[rCnt];
				while((sCnt < sSet.length) && (sSet[sCnt] < element))
					sCnt++;
				
				if((sCnt >= sSet.length) || (sSet[sCnt] != element)) {
					// element from r cannot be found in s
					return false;
				}
				
				sCnt++;
			}
		} else if(sOrder == SORT_ASC) {
			int sCnt = limit;
			for(int rCnt = limit; rCnt < rSet.length; rCnt++) {
				int element = rSet[rCnt];
				while((sCnt < sSet.length) && (sSet[sCnt] > element))
					sCnt++;
				
				if((sCnt >= sSet.length) || (sSet[sCnt] != element)) {
					// element from r cannot be found in s
					return false;
				}
				
				sCnt++;
			}
		}
		return true;
	}

    public void setSortOrder(SortOrder sortOrder) {
        this.sortOrder = sortOrder;
    }

}
