package scj.algorithm.limitcomplete;

import gnu.trove.iterator.TIntIterator;
import gnu.trove.list.array.TIntArrayList;
import it.unimi.dsi.fastutil.ints.IntListIterator;

import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import scj.algorithm.ConfigurableRawDataAlgorithm;
import scj.algorithm.RawDataAlgorithm;
import scj.algorithm.limitremade.structures.LIMITAPrefixTree;
import scj.algorithm.limitremade.structures.LIMITAPrefixTreeNode;
import scj.algorithm.limitremade.structures.LIMITOPJInvertedIndex;
import scj.algorithm.limitremade.structures.TupleIDList;
import scj.algorithm.order.FrequencyOrder;
import scj.algorithm.order.InverseFrequencyOrder;
import scj.algorithm.order.SortOrder;
import scj.algorithm.tree.Node;
import scj.input.DataTuple;
import scj.result.Result;
import scj.runtime.Executable;
import scj.runtime.RuntimeCalculator;

/**
 * Extends the LIMIT algorithm by the join paradigm 'OPJ' which allows
 * to perform the join operation before completely adding the data
 * sets to the tree and index structures.
 * @author Christopher Schiefer
 *
 */
public class LIMITOPJ extends ConfigurableRawDataAlgorithm implements RawDataAlgorithm  {
	private static final Logger LOGGER = LoggerFactory.getLogger(LIMITOPJ.class);

	SortOrder sortOrder;
	static final int SORT_ASC = 0;
	static final int SORT_DESC = 1;
	int sOrder;
	static int limit = 4;
	int[][] rdArray;
	int[][] sdArray;
	long myresult;

	public LIMITOPJ(int limit, String so){
		// set limit
		LIMITOPJ.limit = limit;
		// set sort order
		if(so.equals("ASC")) {
			sOrder = SORT_ASC;
			sortOrder = new InverseFrequencyOrder();
		} else if(so.equals("DESC")) {
			sOrder = SORT_DESC;
			sortOrder = new FrequencyOrder();
		}
	}

	/**
	 * Prepares the data by sorting the tuples according to frequent or
	 * infrequent sort order and transforming the Sets into Arrays.
	 */
	public void preexecute(Set<DataTuple> set1, Set<DataTuple> set2, Result result) {
		sortOrder.initialize(set1);
		for(DataTuple r : set1) {
			r.setSet(sortOrder.sort(r.getSet()));
		}
		// set2 is also ordered for LIMIT
		sortOrder.initialize(set2);
		for(DataTuple r : set2) {
			r.setSet(sortOrder.sort(r.getSet()));
		}
		
		 // this is unnecessary processing but using Arrays is more
		 // efficient with LIMIT and the framework passes Sets.
		rdArray = new int[set1.size()+1][];
						for(DataTuple dt : set1) {
							int[] dtSet = dt.getSet();
							rdArray[dt.getId()] = dtSet;
						}
		sdArray = new int[set2.size()+1][];
						for(DataTuple dt : set2) {
							int[] dtSet = dt.getSet();
							sdArray[dt.getId()] = dtSet;
						}
//		System.out.println("data array built");
	}
	
	/**
	 * Sort dataArray by only rearranging indices pointing on the array.
	 * As a result, we return an array with the pointers arranged in
	 * the correct order of dataArray.
	 */
	public int[] sortRelation(int[][] dataArray) {
		int[] oldIndices = new int[dataArray.length];
		for(int i=1; i<oldIndices.length; i++)
			oldIndices[i] = i;
		
		sortRelationRec(dataArray, oldIndices, 1, dataArray.length-1);
		
		return oldIndices;
	}
	
	/**
	 * Recursive sort step.
	 */
	public void sortRelationRec(int[][] dataArray, int[] oldIndices, int left, int right) {
		if(left >= right)
			return;
		
		swap(dataArray, oldIndices, left, (left+right)/2);
		int last = left;
		int diff, ldiff;
		for(int i=left+1; i <= right; i++) {
			if(sOrder == SORT_ASC) 
				diff = dataArray[i][0] - dataArray[left][0];
			else
				diff = dataArray[left][0] - dataArray[i][0];
			ldiff = dataArray[i].length - dataArray[left].length;
			if((diff > 0) || (diff==0) && ((ldiff < 0) || (ldiff == 0) && (oldIndices[i] < oldIndices[left]))) {
				last = last+1;
				swap(dataArray, oldIndices, last, i);
			}
		}
		swap(dataArray, oldIndices, left, last);
		sortRelationRec(dataArray, oldIndices, left, last-1);
		sortRelationRec(dataArray, oldIndices, last+1, right);
	}
	
	/**
	 *  swaps both elements in the data as well as in the indices array
	 */
	private void swap(int[][] dataArray, int[] oldIndices, int left, int right) {
		int oldLeft = oldIndices[left];
		int[] tData = dataArray[left];
		
		dataArray[left] = dataArray[right];
		oldIndices[left] = oldIndices[right];
		
		dataArray[right] = tData;
		oldIndices[right] = oldLeft;
	}

	
	@Override
	public void execute(final Set<DataTuple> set1, final Set<DataTuple> set2, Result result) {
		System.out.println("LIMIT(OPJ) with limit="+limit);

		RuntimeCalculator rc = new RuntimeCalculator(this.getClass());
		
		final int[][] rArray = rdArray;
		final int[][] sArray = sdArray;
		
		// in contrast to the LIMIT proposed in the original paper, we build the complete
		// PrefixTree ahead and only choose the corresponding subtree.
		LIMITAPrefixTree tr = 
			(LIMITAPrefixTree) rc.measure(new Executable() {
				public Object execute() {
					return new LIMITAPrefixTree(rArray, limit);
				}
			}, "build prefix tree");
//		System.out.println("prefixtree built");
		
		
		int[] oldIndices = 
				(int[]) rc.measure(new Executable() {
					public Object execute() {
						return sortRelation(sArray);
					}
				}, "sort S");
//		System.out.println("S sorted");
		
		//System.out.println("recs in subtree of root: "+tr.getRecsInSubtreeCnt());
		//System.out.println("tree size: "+tr.getSize());
		//System.out.println("tuples included: "+tr.getRecordCnt()+", cutoff: "+tr.getCutOffCnt());

		LIMITOPJInvertedIndex is = 
				(LIMITOPJInvertedIndex) rc.measure(new Executable() {
					public Object execute() {
						return new LIMITOPJInvertedIndex();
					}
				}, "build inv index");
//		System.out.println("inverted index built");
		
		// set pointer to first position of partition
		int partBegin = 1;
		// runs through the the S data array
		int sid = 1;
		
		// determine max value in domain:
		int maxVal;
		if(sOrder == SORT_ASC) {
			maxVal = sArray[1][0];
		} else {
			maxVal = sArray[sArray.length-1][0];
		}
		// loop could possibly be saved with a smarter solution
		for(int i=1; i<rArray.length; i++) {
			if(rArray[i][0] > maxVal)
				maxVal = rArray[i][0];
		}
		
		// distinguish between the sort orders:
		if(sOrder == SORT_ASC) {
			// Ascending/Infrequent sort order
			for(int kid=maxVal; kid>=0; kid--) {
				// increase sid as long as first element of considered tuple is kid
				while((sid < sArray.length) && (sArray[sid][0] == kid)) {
					sid++;
				}
				// check if partition contains new items, otherwise we don't need to
				// update the index
				if(partBegin != sid) {
					// fill index with new partition
					is.fillIndex(sArray, partBegin, sid);
					partBegin = sid;
				}
					
				// choose tree with considered item as its root
				LIMITAPrefixTreeNode c = (LIMITAPrefixTreeNode)tr.getChild(kid);
				if(c != null) { // are there tuples with kid as first element in R?
					this.processNode((LIMITAPrefixTreeNode) c, is , result, limit, rArray, sArray, oldIndices);
				}
			}
		} else if(sOrder == SORT_DESC) {
			// Descending/Frequent sort order
			for(int kid=sArray[1][0]; kid<=maxVal; kid++) {
				// increase sid as long as first element of considered tuple is kid
				while((sid < sArray.length) && (sArray[sid][0] == kid)) {
					sid++;
				}
				// check if partition contains new items, otherwise we don't need to
				// update the index
				if(partBegin != sid) {
					// fill index with new partition
					is.fillIndex(sArray, partBegin, sid);
					partBegin = sid;
				}

				// choose tree with considered item as its root
				LIMITAPrefixTreeNode c = (LIMITAPrefixTreeNode)tr.getChild(kid);
				if(c != null) { // are there tuples with kid as first element in R?
					this.processNode((LIMITAPrefixTreeNode) c, is , result, limit, rArray, sArray, oldIndices);
				}
			}
		}

		LOGGER.info("{}", rc);
	}
	
	/**
	 * Initial processNode() step for the children of the root node, which
	 * does not require an intersection. (Intersection would be done with
	 * one list being the complete set of all tuples from S.)
	 */
	public void processNode(LIMITAPrefixTreeNode currentNode, LIMITOPJInvertedIndex invertedIndex, Result result, int limit, int[][] rArray, int[][] sArray, int[] oldSIndices) {
		TIntArrayList tuples = invertedIndex.get(currentNode.getName());

		// save further steps by checking for empty result
		if(tuples.size() == 0)
			return;

		// add tuples ending in this node to results
		verifySmart(currentNode, tuples, limit, rArray, sArray, oldSIndices, result);

		// check all children nodes in recursive step
		for(Node c : currentNode.getChildren())
			this.processNodeRecursion((LIMITAPrefixTreeNode) c, tuples , invertedIndex , result, limit, rArray, sArray, oldSIndices);
	}
		
	/**
	 * Recursion step for ProcessNode(), which performs an intersection,
	 * adds and verifies tuples to the result list, and calls further 
	 * recursion steps by traversing the subtree from currentNode.
	 */
	public void processNodeRecursion(LIMITAPrefixTreeNode currentNode, TIntArrayList tupleIDCandidates, LIMITOPJInvertedIndex invertedIndex, Result result, int limit, int[][] rArray, int[][] sArray, int[] oldSIndices) {
		// perform intersection of tupleIDCandidates with the list of tuples found
		// in the index for the considered item in this step, as found in currentNode's 
		// root.
		TIntArrayList tuples = TupleIDList.intersectHybrid(
			tupleIDCandidates,
			invertedIndex.get(currentNode.getName()),
			sArray
		);

		// save further steps by checking for empty result
		if(tuples.size() == 0)
			return;
		
		// add tuples ending in this node to results,
		// and potentially check cut off tuples if this is a leaf node.
		verifySmart(currentNode, tuples, limit, rArray, sArray, oldSIndices, result);
		
		// check all children nodes as recursion step
		for(Node c : currentNode.getChildren())
			this.processNodeRecursion((LIMITAPrefixTreeNode) c, tuples , invertedIndex , result, limit, rArray, sArray, oldSIndices);
	}
	
	/**
	 * Verify tuples in this node.
	 */
	public void verifySmart(LIMITAPrefixTreeNode currentNode, TIntArrayList tupleIDCandidates, int safe, int[][] rArray, int[][] sArray, int[] oldSIndices, Result result) {
		TIntIterator it = tupleIDCandidates.iterator();
		// run through candidate list
		while(it.hasNext()) {
			int s = it.next();
			// are there tuples in this node?
			if(currentNode.containsTuples()) {
				IntListIterator itr = currentNode.getTupleIds().iterator();
				while (itr.hasNext()){
					// simply add all pairs to the result list, since we know
					// from the tree traversal that they are correct.
					int r = itr.next();
					result.add(r, oldSIndices[s]);
				}
			}
			// are there cut off tuples to verify?
			if(currentNode.containsCutOffs()) {
				IntListIterator itr = currentNode.getCutOffTuples().iterator();
				int[] sSet = sArray[s];
				while (itr.hasNext()){
					int r = itr.next();
					int[] rSet = rArray[r];
					// we need to check if rSet is included in sSet, as we could not
					// check all items in the tree traversal.
					if(qualify(rSet, sSet, safe)) {
						result.add(r, oldSIndices[s]);
					}
				}
			}
		}
	}
	
	/**
	 * Checks if all items in rSet are contained in sSet. Ignores the first
	 * (or last, depending on the sort order) items, as these were already
	 * checked implicitly by traversing the tree.
	 */
	public boolean qualify(int[] rSet, int[] sSet, int limit) {
		// distinguish between sort order:
		if(sOrder == SORT_DESC) {
			int sCnt = limit;
			// run through complete rSet
			for(int rCnt = limit; rCnt < rSet.length; rCnt++) {
				int element = rSet[rCnt];
				// run through sSet until we find the item from rSet, or
				// exceed sSet's length
				while((sCnt < sSet.length) && (sSet[sCnt] < element))
					sCnt++;
				
				if((sCnt >= sSet.length) || (sSet[sCnt] != element)) {
					// if we stopped the last loop after not finding the item,
					// we know that the tuple rSet is not fully contained in sSet
					return false;
				}
				
				sCnt++;
			}
		} else if(sOrder == SORT_ASC) {
			int sCnt = limit;
			// run through complete rSet
			for(int rCnt = limit; rCnt < rSet.length; rCnt++) {
				int element = rSet[rCnt];
				// run through sSet until we find the item from rSet, or
				// exceed sSet's length
				while((sCnt < sSet.length) && (sSet[sCnt] > element))
					sCnt++;
				
				if((sCnt >= sSet.length) || (sSet[sCnt] != element)) {
					// if we stopped the last loop after not finding the item,
					// we know that the tuple rSet is not fully contained in sSet
					return false;
				}
				
				sCnt++;
			}
		}
		// we found all items from rSet in sSet
		return true;
	}

	/**
	 * Relict of an old implementation, NOT used anymore!
	 * Use the parameter 'so' in the constructor instead.
	 */
    public void setSortOrder(SortOrder sortOrder) {
        this.sortOrder = sortOrder;
    }

}
