#include "SuffixDiskPagePool.hpp"

/**
 * Schreibt count Bytes ab source in den Suffix-Puffer
 */
void SuffixDiskPage::write(char * source, unsigned int count){
	//cerr << "DiskPage::write() begin" << endl;
	//cerr << "put_ptr == " << (long)put_ptr << ", source == " << (long)source << ", count == " << count << endl;
	//cerr << endl;
	memcpy(put_ptr,source, count);
	put_ptr+=count;
	pos_put+=count;
	pos_end = (pos_put > pos_end?pos_put:pos_end);
	dirty = true;
	//cerr << "DiskPage::write() end" << endl;
}


#ifdef USE_C_FILES
/**
 * Schreibt die gesamte Speicherseite auf die Platte
 * (schreibt aber nur so viel wie nötig, da das Ende des Files noch gebraucht wird...)
 *
 */
void SuffixDiskPage::writePage(FILE * & c_file){
	fseek(c_file, pos_start, SEEK_SET);
	fwrite(mPage, 1, (pos_end - pos_start), c_file);
	dirty = false;
}

/**
 * Liest möglichst einen ganzen Block ein. Falls die letzte Seite gelesen wird,
 * werden nicht SUFFIX_PAGESIZE_ON_DISK chars gelesen; der tatsächliche Wert wird abgefragt
 *
 */
void SuffixDiskPage::readPage(FILE * & c_file){
	fseek(c_file, pos_start, SEEK_SET);
	pos_end = pos_start + fread(mPage, 1, nsSuffixPageSize, c_file); // fwrite gibt die tatsächlich geschriebenen Bytes zurück
	dirty = false;
}
#else
/**
 * Schreibt die gesamte Speicherseite auf die Platte
 * (schreibt aber nur so viel wie nötig, da das Ende des Files noch gebraucht wird...)
 *
 */
void SuffixDiskPage::writePage(fstream& stream){
	stream.clear();
	stream.seekp(pos_start);
	stream.write(mPage, (pos_end - pos_start));
	dirty = false;
}

/**
 * Liest möglichst einen ganzen Block ein. Falls die letzte Seite gelesen wird,
 * werden nicht SUFFIX_PAGESIZE_ON_DISK chars gelesen; der tatsächliche Wert wird abgefragt
 *
 */
void SuffixDiskPage::readPage(fstream& stream){
	stream.seekg(pos_start);
	stream.read(mPage, nsSuffixPageSize); // möglichst alles lesen
	pos_end = pos_start + stream.gcount();// tatsächlich eingelesene Bytes
	stream.clear();
	dirty = false;
}
#endif

/**
 * D'tor: schaut in der map nach, ob noch Seiten drin sind
 * Seite dirty --> schreiben und löschen, sonst nur löschen
 */
SuffixDiskPagePool::~SuffixDiskPagePool(){
	for(SuffixPagePoolBufferIterator it = pagePool.begin(); it != pagePool.end(); ++it){
		SuffixDiskPage* dp = (*it).second;
		if(dp->isDirty()){
#ifdef USE_C_FILES
			dp->writePage(c_suffixFile);
#else
			dp->writePage(suffixFile);
#endif
		}
		delete dp;
	}
}

/**
 * Setzt den put-Pointer und bereitet den direkten Zugriff auf die Page vor
 * 1. Position setzen
 * 2. Seite suchen und ggf. Laden
 * 3. Zeiger auf aktuelle Seite setzen (currentPutPage)
 *
 */
void SuffixDiskPagePool::seekp(POS_TYPE pos){
	pos_put = pos;
	// startPos berechnen
	POS_TYPE startPos = pos_put / nsSuffixPageSize;
	startPos = (POS_TYPE)startPos * (POS_TYPE)nsSuffixPageSize;

	currentPutPage = getFromPool(startPos);
	// spart einen Funktionsaufruf
	if(lruList.latest != startPos)lruList.touch(startPos);

	if(currentPutPage == NULL){
		currentPutPage = new SuffixDiskPage(startPos);
#ifdef USE_C_FILES
		currentPutPage->readPage(c_suffixFile);
#else
		currentPutPage->readPage(suffixFile);
#endif
		insertIntoPool(currentPutPage);
	}
	currentPutPage->seekp(pos);
}

/**
 * Setzt den get-Pointer und bereitet den direkten Zugriff auf die Page vor<br/>
 * 1. Position setzen<br/>
 * 2. Seite suchen und ggf. Laden<br/>
 * 3. Zeiger auf aktuelle Seite setzen (currentGetPage)<br/>
 *
 */
void SuffixDiskPagePool::seekg(POS_TYPE pos){
	pos_get = pos;
	// startPos berechnen
	POS_TYPE startPos = pos_get / nsSuffixPageSize;

	startPos = (POS_TYPE)startPos * (POS_TYPE)nsSuffixPageSize;
	// spart einen Funktionsaufruf
	if(lruList.latest != startPos)lruList.touch(startPos);

	if(special_debug)streamout(cerr, startPos);
	if(special_debug)streamout(cerr, pos);

	currentGetPage = getFromPool(startPos);
	if(currentGetPage == NULL){
		currentGetPage = new SuffixDiskPage(startPos);
#ifdef USE_C_FILES
		currentGetPage->readPage(c_suffixFile);
#else
		currentGetPage->readPage(suffixFile);
#endif
		insertIntoPool(currentGetPage);
	}
	currentGetPage->seekg(pos);

}

/**
 * sucht die SuffixDiskPage aus dem Pool heraus, sofern vorhanden
 */
SuffixDiskPage * SuffixDiskPagePool::getFromPool(POS_TYPE startPos){
	SuffixDiskPage * ret = NULL;
	SuffixPagePoolBufferIterator it = pagePool.find(startPos);
	if(it != pagePool.end()){
		ret = it->second;
	}
	return ret;
}

/**
 * Fügt eine Seite in den Cache ein
 */
bool SuffixDiskPagePool::insertIntoPool(SuffixDiskPage * page){
	if(nsSuffixPagesInBuffer == pagePool.size()){
		//cerr << "NUMBER_OF_SUFFIX_DISKPAGES_IN_MEMORY == pagePool.size()" << endl;
		// Pool ist voll; aus der lru-Liste die eine Seite herausnehmen
		POS_TYPE pagePosToReplace = lruList.pop_oldest();
		SuffixDiskPage * pageToReplace = pagePool.find(pagePosToReplace)->second;
		// kann an der Stelle einfach nicht null sein, daher auch keine Überprüfung
		// wenn in LRUList, dann auch im PagePool
		pagePool.erase(pagePosToReplace);
		if(pageToReplace->isDirty()){
#ifdef USE_C_FILES
			pageToReplace->writePage(c_suffixFile);
#else
			pageToReplace->writePage(suffixFile);
#endif
		}
		delete pageToReplace;
	}
	pagePool.insert(pair<POS_TYPE, SuffixDiskPage*>(page->getStartPos(), page));
	return true;
}

