package org.sidiff.common.util;

import java.io.Serializable;
import java.util.*;

/**
 * Utility class for counting things or measuring times.
 * @author wenzel / reuling
 */
public class StatisticsUtil implements Serializable {	
	
	/**
	 * 
	 */
	private static final long serialVersionUID = -6455875097937890547L;
	
	private static final String LINE_SEPERATOR = System.getProperty("line.separator");

	public enum StatisticType {
		Time,Size,Count,Other;
	}

	private static final String STAT_KEY_STARTTIME = "@@STARTOF@@";	

	private HashMap<String, Object> timeStatistic;
	private HashMap<String, Object> sizeStatistic;
	private HashMap<String, Object> countStatistic;
	private HashMap<String, Object> otherStatistic;

	private static StatisticsUtil instance;
	private static boolean enabled = true;
	
	private StatisticsUtil() {
		this.timeStatistic = new HashMap<String, Object>();
		this.sizeStatistic = new HashMap<String, Object>();
		this.countStatistic = new HashMap<String, Object>();
		this.otherStatistic = new HashMap<String, Object>();
	}
	
	@SuppressWarnings("unchecked")
	private StatisticsUtil(HashMap<String, Object> timeStatistic, HashMap<String, Object>sizeStatistic, HashMap<String, Object> countStatistic, 
			HashMap<String, Object> otherStatistic) {
		this.timeStatistic = (HashMap<String, Object>) timeStatistic.clone();
		this.sizeStatistic = (HashMap<String, Object>) sizeStatistic.clone();
		this.countStatistic = (HashMap<String, Object>) countStatistic.clone();
		this.otherStatistic = (HashMap<String, Object>) otherStatistic.clone();
	}
	
	public static StatisticsUtil getInstance() {
		if (instance == null)
			instance = new StatisticsUtil();
		return instance;
	}
	
	/**
	 * Copy constructor, using a static method.
	 */
	public static StatisticsUtil copiedInstance(StatisticsUtil statisticsUtil) {
		return new StatisticsUtil(statisticsUtil.getTimeStatistic(), statisticsUtil.getSizeStatistic(), statisticsUtil.getCountStatistic(), statisticsUtil.getOtherStatistic());
	}
	
	/**
	 * Disables the StatisticsUtil, used for performance reason
	 */
	public static void disable(){
		enabled = false;
	}

	/**
	 * Reenables the StatisticsUtil
	 */
	public static void reenable(){
		enabled = true;
	}
	
	public HashMap<String, Object> getTimeStatistic() {
		return timeStatistic;
	}

	public HashMap<String, Object> getSizeStatistic() {
		return sizeStatistic;
	}

	public HashMap<String, Object> getCountStatistic() {
		return countStatistic;
	}

	public HashMap<String, Object> getOtherStatistic() {
		return otherStatistic;
	}

	/**
	 * Resets the StatisticsUtil. All data will be removed.
	 */
	public void reset() {
		if(enabled){
		this.timeStatistic.clear();
		this.sizeStatistic.clear();
		this.countStatistic.clear();
		this.otherStatistic.clear();
		}
	}
	
	/**
	 * Resets all counters whose keys start with the given prefix. 
	 * @param keyPrefix
	 */
	public void resetCounters(String keyPrefix) {
		if (enabled) {
			reset(keyPrefix, countStatistic);
		}
	}
	
	/**
	 * Resets all sizes (integer values) whose keys start with the given prefix. 
	 * @param keyPrefix
	 */
	public void resetSizes(String keyPrefix) {
		if (enabled) {
			reset(keyPrefix, sizeStatistic);
		}
	}
	
	/**
	 * Resets all times whose keys start with the given prefix. 
	 * @param keyPrefix
	 */
	public void resetTimes(String keyPrefix) {
		if (enabled) {
			reset(keyPrefix, timeStatistic);
		}
	}
	
	/**
	 * Resets other statistics whose keys start with the given prefix. 
	 * @param keyPrefix
	 */
	public void resetOthers(String keyPrefix) {
		if (enabled) {
			reset(keyPrefix, otherStatistic);
		}
	}
	
	/**
	 * Resets all stored values whose keys start with the given prefix. 
	 * @param keyPrefix
	 */
	public void reset(String keyPrefix) {
		if (enabled) {
			resetCounter(keyPrefix);
			resetSizes(keyPrefix);
			resetTime(keyPrefix);
			resetOthers(keyPrefix);
		}		
	}
	
	/**
	 * Resets all statistics whose keys start with the given prefix. 
	 * 
	 * @param keyPrefix
	 * @param statistic
	 */
	private void reset(String keyPrefix, Map<String, Object> statistic) {
		if (enabled) {
			Set<String> keys = new HashSet<String>();
			
			for (String key : statistic.keySet()) {
				if (key.startsWith(keyPrefix)) {
					keys.add(key);
				}
			}
			
			for (String key : keys) {
				statistic.remove(key);
			}
		}
	}

	/**
	 * Starts the measurement of time for a given key.
	 * @param key
	 */
	public void start(String key) {
		if (enabled) {
			Long time = (Long) timeStatistic.get(key);
			if (time == null)
				time = 0l;
			timeStatistic.put(STAT_KEY_STARTTIME + key, new Long(System.currentTimeMillis() - time));
		}
	}

	/**
	 * Stop the measurement of time for a given key. The time difference between the 
	 * start time and the stop time will be stored.
	 * @param key
	 * @return
	 */
	public float stop(String key) {
		if (enabled) {
			long stop = System.currentTimeMillis();
			Long startO = (Long) timeStatistic.get(STAT_KEY_STARTTIME + key);
			long start = startO == null ? 0 : startO.longValue();
			long time = (stop - start);
			timeStatistic.remove(STAT_KEY_STARTTIME + key);
			timeStatistic.put(key, time);
			return time / 1000f;
		}
		return 0;
	}

	/**
	 * Removes the measurement of time for a given key. 
	 * @param key
	 */
	public void resetTime(Object key) {
		if(enabled){
		timeStatistic.remove(STAT_KEY_STARTTIME + key);
		}
	}
	
	/**
	 * Returns the time that has been measured for the given key.
	 * @param key
	 * @return
	 */
	public float getTime(Object key) {
		if (enabled) {
			try {
				return ((Long) timeStatistic.get(key)) / 1000f;
			} catch (Exception e) {
				return -1f;
			}
		}
		return -1f;
	}

	/**
	 * Returns the data that is stored for the given key.
	 * @param key
	 * @return
	 */
	public Object getObject(String key) {
		if (enabled) {
			return otherStatistic.get(key);
		}
		return null;
	}

	/**
	 * Returns the string data that is stored for the given key.
	 * @param key
	 * @return
	 */
	public String getString(String key) {
		if (enabled) {
			return (String) otherStatistic.get(key);
		}
		return null;
	}

	/**
	 * Returns the integer data that is stored for the given key.
	 * @param key
	 * @return
	 */
	public int getInt(String key) {
		if (enabled) {
			Integer i = ((Integer) otherStatistic.get(key));
			return (i == null) ? 0 : i.intValue();
		}
		return 0;
	}

	/**
	 * Stores an arbitrary data value for the given key.
	 * @param key
	 * @param value
	 */
	public void put(String key, Object value) {
		if (enabled) {
			otherStatistic.put(key, value);
		}
	}

	/**
	 * Stores an integer value for the given key.
	 * @param key
	 * @param value
	 */
	public void put(String key, int value) {
		if (enabled) {
			otherStatistic.put(key, new Integer(value));
		}
	}

	/**
	 * Stores a size (integer value) of a given key.
	 * @param key
	 * @param value
	 */
	public void putSize(Object key, int value) {
		if (enabled) {
			sizeStatistic.put((String) key, value);
		}
	}

	/**
	 * Returns the size (integer value) of a given key.
	 * @param key
	 * @return
	 */
	public int getSize(Object key) {
		if(enabled){
		return (Integer) sizeStatistic.get(key);
		}
		return 0;
	}
	
	/**
	 * Returns the counter of the given key.
	 * @param key
	 * @return
	 */
	public int getCounter(Object key) {
		if (enabled) {
			return (Integer) countStatistic.get(key);
		}
		return 0;
	}
	
	/**
	 * Resets the counter of the given key.
	 * @param key
	 */
	public void resetCounter(Object key) {
		if (enabled) {
			countStatistic.put((String) key, new Integer(0));
		}
	}

	/**
	 * Increases the counter for the given key.
	 * @param key
	 */
	public void count(Object key) {
		if (enabled) {
			int i = (Integer) countStatistic.get(key);
			countStatistic.put((String) key, ++i);
		}
	}	
	
	/**
	 * Returns a textual dump of the stored information.
	 * @return
	 */
	public String dump() {
		if (enabled) {
			String string = new String("*********************" + "Statistics" + "*********************" + LINE_SEPERATOR);
			string += "Time Statistics(in ms):" + LINE_SEPERATOR;
			ArrayList<String> keys = new ArrayList<String>(timeStatistic.keySet());
			Collections.sort(keys);
			for (String key : keys)
				string += key + " = " + timeStatistic.get(key) + LINE_SEPERATOR;
			string += LINE_SEPERATOR;
			string += "Count statistics:" + LINE_SEPERATOR;
			ArrayList<String> keys2 = new ArrayList<String>(countStatistic.keySet());
			Collections.sort(keys2);
			for (String key : keys2)
				string += key + " = " + countStatistic.get(key) + LINE_SEPERATOR;
			string += LINE_SEPERATOR;
			string += "Size statistics:" + LINE_SEPERATOR;
			ArrayList<String> keys3 = new ArrayList<String>(sizeStatistic.keySet());
			Collections.sort(keys3);
			for (String key : keys3)
				string += key + " = " + sizeStatistic.get(key) + LINE_SEPERATOR;
			string += LINE_SEPERATOR;
			string += "Other statistics:" + LINE_SEPERATOR;
			ArrayList<String> keys4 = new ArrayList<String>(otherStatistic.keySet());
			Collections.sort(keys4);
			for (String key : keys4)
				string += key + " = " + otherStatistic.get(key) + LINE_SEPERATOR;
			return string + "**************************************************************";
		}
		return "";
	}
	
}
