package org.sidiff.difference.rulebase.wrapper;

import static org.sidiff.common.henshin.HenshinRuleAnalysisUtilEx.isAmalgamationUnit;
import static org.sidiff.common.henshin.HenshinRuleAnalysisUtilEx.isKernelRule;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.henshin.model.Attribute;
import org.eclipse.emf.henshin.model.ConditionalUnit;
import org.eclipse.emf.henshin.model.IndependentUnit;
import org.eclipse.emf.henshin.model.IteratedUnit;
import org.eclipse.emf.henshin.model.LoopUnit;
import org.eclipse.emf.henshin.model.Module;
import org.eclipse.emf.henshin.model.Node;
import org.eclipse.emf.henshin.model.PriorityUnit;
import org.eclipse.emf.henshin.model.Rule;
import org.eclipse.emf.henshin.model.SequentialUnit;
import org.eclipse.emf.henshin.model.Unit;
import org.sidiff.common.henshin.EditRuleAnnotations;
import org.sidiff.common.henshin.HenshinModuleAnalysis;
import org.sidiff.difference.rulebase.RuleBaseItem;
import org.sidiff.difference.symmetric.SymmetricPackage;
import org.silift.common.util.emf.EMFStorage;

public class RuleBaseItemInfo {
	
	public static String getName(RuleBaseItem item) {
		return item.getEditRule().getExecuteModule().getName();
	}
	
	public static void setName(RuleBaseItem item, String name) {
		if (!name.equals(getName(item))) {
			item.getEditRule().getExecuteModule().setName(name);
		}
	}

	public static String getDescription(RuleBaseItem item) {
		return item.getEditRule().getExecuteModule().getDescription();
	}
	
	public static void setDescription(RuleBaseItem item, String name) {
		if (!name.equals(getDescription(item))) {
			item.getEditRule().getExecuteModule().setDescription(name);
		}
	}

	public static String formatName(RuleBaseItem item) {
		String internalName = getName(item);
		
		// Remove quotes:
		String displayName = internalName.substring(1, internalName.length() - 1);
		
		// Remove underscore
		displayName = displayName.replace('_', ' ');

		// Remove camel-case
		String regex = "([a-z])([A-Z]+)";
        String replacement = "$1 $2";
        displayName = displayName.replaceAll(regex, replacement);
		
        // Make first letters upper-case
        displayName = capitalizeFirstLetter(displayName);
        
        // Translate special words:
        displayName = translate(displayName);
        
        return displayName;
	}

	private static String capitalizeFirstLetter(String input) {
		StringBuilder result = new StringBuilder(input.length());
		String[] words = input.split("\\s");
		
		for (int i = 0, l = words.length; i < l; ++i) {
			if (i > 0) {
				result.append(" ");
			}
			result.append(Character.toUpperCase(words[i].charAt(0)))
					.append(words[i].substring(1));

		}
		return result.toString();
	}
	
	private static String translate(String input) {
		StringBuilder result = new StringBuilder(input.length());
		String[] words = input.split("\\s");
		
		for (int i = 0, l = words.length; i < l; ++i) {
			if (i > 0) {
				result.append(" ");
			}
			result.append(dictionary(words[i]));

		}
		return result.toString();
	}
	
	private static final Map<String, String> dict;
	static {
		dict = new HashMap<String, String>();
		dict.put("SET", "Set");
		dict.put("UNSET", "Unset");
		dict.put("ADD", "Add");
		dict.put("CREATE", "Create");
		dict.put("DELETE", "Delete");
		dict.put("MOVE", "Move");
		dict.put("REMOVE", "Remove");
		dict.put("CHANGE", "Change");
		dict.put("NOT", "Not");
		dict.put("REFERENCE", "Reference");
		dict.put("MOVEs", "Moves");
		dict.put("CHANGEs", "Changes");
		dict.put("FROM", "From");
		dict.put("TO", "To");
		dict.put("IN", "In");
		dict.put("ATTRIBUTE", "Attribute");
		dict.put("Id", "ID");
		dict.put("TGT", "Target");
		dict.put("SRC", "Source");
	}

	private static String dictionary(String input) {
		// Translate:
		String output = dict.get(input);
		
		if (output == null) {
			return input;
		} else {
			return output;
		}
	}
	
	public static int getRefinementLevel(RuleBaseItem item) {
		return Integer.valueOf(getSematicChangeSetRefinementLevel(item).getValue());
	}
	
	public static int getNumberOfACs(RuleBaseItem item) {
		if (getSematicChangeSetNumberOfACs(item) == null){
			// Compatibility
			return 0;
		} else {
			return Integer.valueOf(getSematicChangeSetNumberOfACs(item).getValue());
		}		
	}
	
	public static int getNumberOfParams(RuleBaseItem item) {
		if (getSematicChangeSetNumberOfParams(item) == null){
			// Compatibility
			return 0;
		} else {
			return Integer.valueOf(getSematicChangeSetNumberOfParams(item).getValue());
		}	
	}

	public static boolean isActiv(RuleBaseItem item) {
		return item.isActive();
	}

	public static void setActive(RuleBaseItem item, boolean value) {
		item.setActive(value);
	}

	public static void invertActivity(RuleBaseItem item) {
		if (item.isActive()) {
			setActive(item, false);
		} else {
			setActive(item, true);
		}
	}

	public static int getPriority(RuleBaseItem item) {
		Integer priority = EditRuleAnnotations.getPriority(item.getEditRule().getExecuteModule());

		if (priority == null) {
			// Return Recognition-Rule (default) value:
			priority = Integer.valueOf(getSematicChangeSetPriority(item).getValue());
		}
		
		return priority;
	}

	public static void setPriority(RuleBaseItem item, int priority) {
		EditRuleAnnotations.setPriority(item.getEditRule().getExecuteModule(), priority);
	}

	public static EClass getERType(RuleBaseItem item) {
		// Get unit type of execute main unit
		return item.getEditRule().getExecuteMainUnit().eClass();
	}

	public static String getDisplayERType(RuleBaseItem item) {
		// Get unit type of execute main unit
		return getUnitType(item.getEditRule().getExecuteMainUnit());
	}

	public static EClass getRRType(RuleBaseItem item) {
		// Get unit type of recognition main unit
		return item.getRecognitionRule().getRecognitionMainUnit().eClass();
	}

	public static String getDisplayRRType(RuleBaseItem item) {
		// Get unit type of recognition main unit
		return getUnitType(item.getRecognitionRule().getRecognitionMainUnit());
	}

	public static URI getEditRuleURI(RuleBaseItem item) {
		return EMFStorage.getURI(item.getEditRule().getExecuteMainUnit());
	}

	public static URI getRecognitionRuleURI(RuleBaseItem item) {
		return EMFStorage.getURI(item.getRecognitionRule().getRecognitionMainUnit());
	}

	/**
	 * Returns the unit type of the given unit.
	 * 
	 * @param unit
	 *            the transformation unit instance.
	 * @return the unit type.
	 */
	private static String getUnitType(Unit unit) {
		
		if (unit instanceof Rule) {
			if (isKernelRule((Rule) unit)) {
				return "Amalgamation Rule";
			} else {
				return "Rule";
			}
		}

		else if (isAmalgamationUnit(unit)) {
			return "Amalgamation Unit";
		}
		
		else if (unit instanceof IndependentUnit) {
			return "Independent";
		}

		else if (unit instanceof SequentialUnit) {
			return "Sequential";
		}

		else if (unit instanceof LoopUnit) {
			return "Loop";
		}
		
		else if (unit instanceof IteratedUnit) {
			return "Iterated";
		}

		else if (unit instanceof ConditionalUnit) {
			return "Conditional";
		}

		else if (unit instanceof PriorityUnit) {
			return "Priority";
		}
		return null;
	}
	
	private static Attribute getSematicChangeSetPriority(RuleBaseItem item) {

		for (Attribute attribute : getSematicChangeSet(item).getAttributes()) {
			if (attribute.getType() == SymmetricPackage.eINSTANCE
					.getSemanticChangeSet_Priority()) {
				return attribute;
			}
		}

		return null;
	}

	private static Attribute getSematicChangeSetRefinementLevel(RuleBaseItem item) {

		for (Attribute attribute : getSematicChangeSet(item).getAttributes()) {
			if (attribute.getType() == SymmetricPackage.eINSTANCE
					.getSemanticChangeSet_RefinementLevel()) {
				return attribute;
			}
		}

		return null;
	}
	
	private static Attribute getSematicChangeSetNumberOfACs(RuleBaseItem item) {

		for (Attribute attribute : getSematicChangeSet(item).getAttributes()) {
			if (attribute.getType() == SymmetricPackage.eINSTANCE
					.getSemanticChangeSet_NumberOfACs()) {
				return attribute;
			}
		}

		return null;
	}
	
	private static Attribute getSematicChangeSetNumberOfParams(RuleBaseItem item) {

		for (Attribute attribute : getSematicChangeSet(item).getAttributes()) {
			if (attribute.getType() == SymmetricPackage.eINSTANCE
					.getSemanticChangeSet_NumberOfParams()) {
				return attribute;
			}
		}

		return null;
	}
	
	private static Node getSematicChangeSet(RuleBaseItem item) {
		Module recognitionModule = item.getRecognitionRule().getRecognitionModule();

		for (Rule rule : HenshinModuleAnalysis.getAllRules(recognitionModule)) {
			for (Node node : rule.getRhs().getNodes()) {
				if (node.getType() == SymmetricPackage.eINSTANCE.getSemanticChangeSet()) {
					return node;
				}
			}
		}

		return null;
	}
}
