package org.sidiff.difference.ensure.evaluation.engine;

import java.util.ArrayList;
import java.util.List;

import org.sidiff.difference.asymmetric.facade.util.Difference;
import org.sidiff.difference.ensure.evaluation.utils.EvaluationUtil;
import org.sidiff.difference.evaluation.AbstractTestCaseEntry;
import org.sidiff.difference.evaluation.ConcreteTestCaseEntry;
import org.sidiff.difference.evaluation.Header;
import org.sidiff.difference.evaluation.engine.AbstractEvaluationEngine;

/**
 * Concrete evaluation engine for our SA/FT PPU case study.
 * 
 * @author kehrer
 */
public class EnsureResultsEvaluationEngine extends AbstractEvaluationEngine {

	@Override
	public List<Header> createHeaders() {
		List<Header> headers = new ArrayList<Header>();
		String[] headerNames = { "#Node", "#Edge", "SUM low-level", "#SERGe_SAFT", "#SERGe_Trace", "SUM SERGe", "#Intra",
				"#Inter", "SUM inter-intra", "#Remaining Low-level", "Reduction1", "Reduction2" };
		for (String headerName : headerNames) {
			Header header = new Header(headerName);
			headers.add(header);
		}
		return headers;
	}

	@Override
	public List<AbstractTestCaseEntry> createTestCaseEntry(List<Difference> differences) {

		super.getEvaluation().setName("Ensure Results");

		List<AbstractTestCaseEntry> entries = new ArrayList<AbstractTestCaseEntry>();

		Header node_header = getEvaluation().getHeader("#Node");
		Header edge_header = getEvaluation().getHeader("#Edge");
		Header sumLowLevel_Header = getEvaluation().getHeader("SUM low-level");
		
		Header sergeSAFT_header = getEvaluation().getHeader("#SERGe_SAFT");
		Header sergeTrace_header = getEvaluation().getHeader("#SERGe_Trace");
		Header sumSerge_header = getEvaluation().getHeader("SUM SERGe");
		
		Header intraModelChange_header = getEvaluation().getHeader("#Intra");
		Header interModelChange_header = getEvaluation().getHeader("#Inter");
		Header sumInterIntra_header = getEvaluation().getHeader("SUM inter-intra");
		Header remainingLowLevel_header = getEvaluation().getHeader("#Remaining Low-level");
		Header reduction1_header = getEvaluation().getHeader("Reduction1");
		Header reduction2_header = getEvaluation().getHeader("Reduction2");

		// Node
		ConcreteTestCaseEntry node_entry = null;
		for (Difference difference : differences) {
			String differenceName = difference.getSymmetric().eResource().getURI().lastSegment();
			if (differenceName.contains("ALL")) {
				node_entry = new ConcreteTestCaseEntry(node_header, EvaluationUtil.getAddRemoveObjects(difference.getSymmetric()).size());
			}
		}

		// Edge
		ConcreteTestCaseEntry edge_entry = null;
		for (Difference difference : differences) {
			String differenceName = difference.getSymmetric().eResource().getURI().lastSegment();
			if (differenceName.contains("ALL")) {
				edge_entry = new ConcreteTestCaseEntry(edge_header, EvaluationUtil.getAddRemoveReferences(difference.getSymmetric()).size());
			}
		}
		
		// SERGe_SAFT
		ConcreteTestCaseEntry sergeSAFT_entry = null;
		for (Difference difference : differences) {
			String differenceName = difference.getSymmetric().eResource().getURI().lastSegment();
			if (differenceName.contains("SERGe")) {
				sergeSAFT_entry = new ConcreteTestCaseEntry(sergeSAFT_header, EvaluationUtil.getSERGe_SAFT_Changes(
						difference.getSymmetric()).size());
			}
		}
		
		// SERGe_Trace
		ConcreteTestCaseEntry sergeTrace_entry = null;
		for (Difference difference : differences) {
			String differenceName = difference.getSymmetric().eResource().getURI().lastSegment();
			if (differenceName.contains("SERGe")) {
				sergeTrace_entry = new ConcreteTestCaseEntry(sergeTrace_header, EvaluationUtil.getSERGe_Trace_Changes(
						difference.getSymmetric()).size());
			}
		}
		
		// Intra
		ConcreteTestCaseEntry intraModelChange_entry = null;
		for (Difference difference : differences) {
			String differenceName = difference.getSymmetric().eResource().getURI().lastSegment();
			if (differenceName.contains("ALL")) {
				intraModelChange_entry = new ConcreteTestCaseEntry(intraModelChange_header, EvaluationUtil.getIntraModelChanges(
						difference.getSymmetric()).size());
			}
		}

		// Inter
		ConcreteTestCaseEntry interModelChange_entry = null;
		for (Difference difference : differences) {
			String differenceName = difference.getSymmetric().eResource().getURI().lastSegment();
			if (differenceName.contains("ALL")) {
				interModelChange_entry = new ConcreteTestCaseEntry(interModelChange_header, EvaluationUtil.getInterModelChanges(
						difference.getSymmetric()).size());
			}
		}

		// Remaining Low-level
		ConcreteTestCaseEntry remainingLowLevel_entry = null;
		for (Difference difference : differences) {
			String differenceName = difference.getSymmetric().eResource().getURI().lastSegment();
			if (differenceName.contains("ALL")) {
				remainingLowLevel_entry = new ConcreteTestCaseEntry(remainingLowLevel_header, EvaluationUtil.getRemainingLowLevelChanges(
						difference.getSymmetric()).size());
			}
		}

		// SUM low-level
		int sumLowLevel_value_value = node_entry.getStats().intValue() + edge_entry.getStats().intValue();				
		ConcreteTestCaseEntry sumLowLevel_entry_entry = new ConcreteTestCaseEntry(sumLowLevel_Header, sumLowLevel_value_value);
		
		// SUM SERGe
		int sumSERGe_value = sergeSAFT_entry.getStats().intValue() + sergeTrace_entry.getStats().intValue();			
		ConcreteTestCaseEntry sumSERGe_entry = new ConcreteTestCaseEntry(sumSerge_header, sumSERGe_value);
		
		// SUM inter-intra
		int sumInterIntra_value = interModelChange_entry.getStats().intValue() + intraModelChange_entry.getStats().intValue();
		ConcreteTestCaseEntry sumInterIntra_entry = new ConcreteTestCaseEntry(sumInterIntra_header, sumInterIntra_value);

		// Reduction 1
		double reduction_value1 = (1 - ((double) sumInterIntra_value / (double) sumLowLevel_value_value)) * 100;
		ConcreteTestCaseEntry reduction1_entry = new ConcreteTestCaseEntry(reduction1_header, reduction_value1);
		
		// Reduction 2
		double reduction_value2 = (1 - ((double) sumInterIntra_value / (double) sumSERGe_value)) * 100;
		ConcreteTestCaseEntry reduction2_entry = new ConcreteTestCaseEntry(reduction2_header, reduction_value2);

		// ... must be in the same order as headers		
		entries.add(node_entry);
		entries.add(edge_entry);
		entries.add(sumLowLevel_entry_entry);
		
		entries.add(sergeSAFT_entry);
		entries.add(sergeTrace_entry);
		entries.add(sumSERGe_entry);
		
		entries.add(intraModelChange_entry);
		entries.add(interModelChange_entry);
		entries.add(sumInterIntra_entry);
		entries.add(remainingLowLevel_entry);
		entries.add(reduction1_entry);
		entries.add(reduction2_entry);

		return entries;
	}

	public String getKey() {
		return "ResultEvaluation";
	}

}
