package org.sidiff.difference.ensure.evaluation.engine;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.sidiff.difference.asymmetric.facade.util.Difference;
import org.sidiff.difference.ensure.evaluation.utils.EvaluationUtil;
import org.sidiff.difference.evaluation.AbstractTestCaseEntry;
import org.sidiff.difference.evaluation.ConcreteTestCaseEntry;
import org.sidiff.difference.evaluation.Header;
import org.sidiff.difference.evaluation.engine.AbstractEvaluationEngine;
import org.sidiff.difference.symmetric.SemanticChangeSet;

/**
 * Concrete evaluation engine for our SA/FT PPU case study.
 * 
 * @author kehrer
 */
public class EnsureDistributionEvaluationEngine extends AbstractEvaluationEngine {

	//private Map<String, Integer> inter = new HashMap<String, Integer>();

	private Map<String, Integer> consistency = new HashMap<String, Integer>();
	private Map<String, Integer> coupling = new HashMap<String, Integer>();
	private Map<String, Integer> decoupling = new HashMap<String, Integer>();
	private Map<String, Integer> propagation = new HashMap<String, Integer>();
	
	
	@Override
	public List<Header> createHeaders() {
		List<Header> headers = new ArrayList<Header>();
		String[] headerNames = { "consistency", "coupling", "decoupling", "propagation" };

		for (String header : headerNames) {
			headers.add(new Header(header));
		}

		
		for (List<Difference> differences : getDifferences()) {
			for (Difference difference : differences) {
				String differenceName = difference.getSymmetric().eResource().getURI().lastSegment();
				if (differenceName.contains("ALL")) {
					for (SemanticChangeSet cs : EvaluationUtil.getConsistencyInterModelChanges(difference.getSymmetric())) {
						if (!consistency.containsKey(cs.getEditRName())) {
							consistency.put(cs.getEditRName(), 0);
						}
					}
					for (SemanticChangeSet cs : EvaluationUtil.getCouplingInterModelChanges(difference.getSymmetric())) {
						if (!coupling.containsKey(cs.getEditRName())) {
							coupling.put(cs.getEditRName(), 0);
						}
					}
					for (SemanticChangeSet cs : EvaluationUtil.getDecouplingInterModelChanges(difference.getSymmetric())) {
						if (!decoupling.containsKey(cs.getEditRName())) {
							decoupling.put(cs.getEditRName(), 0);
						}
					}
					for (SemanticChangeSet cs : EvaluationUtil.getPropagationInterModelChanges(difference.getSymmetric())) {
						if (!propagation.containsKey(cs.getEditRName())) {
							propagation.put(cs.getEditRName(), 0);
						}
					}
				}
			}
		}

		// Create headers
		List<String> consistencyNames = new ArrayList<String>(consistency.keySet());
		Collections.sort(consistencyNames);
		List<String> couplingNames = new ArrayList<String>(coupling.keySet());
		Collections.sort(couplingNames);
		List<String> decouplingNames = new ArrayList<String>(decoupling.keySet());
		Collections.sort(decouplingNames);
		List<String> propagationNames = new ArrayList<String>(propagation.keySet());
		Collections.sort(propagationNames);
		
		for (String er : consistencyNames) {
			Header header = new Header(er);
			headers.get(0).getSubHeaders().add(header);
		}
		for (String er : couplingNames) {
			Header header = new Header(er);
			headers.get(1).getSubHeaders().add(header);
		}
		for (String er : decouplingNames) {
			Header header = new Header(er);
			headers.get(2).getSubHeaders().add(header);
		}
		for (String er : propagationNames) {
			Header header = new Header(er);
			headers.get(3).getSubHeaders().add(header);
		}

		return headers;
	}

	@Override
	public List<AbstractTestCaseEntry> createTestCaseEntry(List<Difference> differences) {

		for (String key : consistency.keySet()) {
			consistency.put(key, 0);
		}
		for (String key : coupling.keySet()) {
			coupling.put(key, 0);
		}
		for (String key : decoupling.keySet()) {
			decoupling.put(key, 0);
		}
		for (String key : propagation.keySet()) {
			propagation.put(key, 0);
		}

		List<AbstractTestCaseEntry> entries = new ArrayList<AbstractTestCaseEntry>();

		for (Difference difference : differences) {
			String differenceName = difference.getSymmetric().eResource().getURI().lastSegment();
			if (differenceName.contains("ALL")) {
				for (SemanticChangeSet cs : EvaluationUtil.getConsistencyInterModelChanges(difference.getSymmetric())) {
					consistency.put(cs.getEditRName(), consistency.get(cs.getEditRName()) + 1);
				}
				for (SemanticChangeSet cs : EvaluationUtil.getCouplingInterModelChanges(difference.getSymmetric())) {
					coupling.put(cs.getEditRName(), coupling.get(cs.getEditRName()) + 1);
				}
				for (SemanticChangeSet cs : EvaluationUtil.getDecouplingInterModelChanges(difference.getSymmetric())) {
					decoupling.put(cs.getEditRName(), decoupling.get(cs.getEditRName()) + 1);
				}
				for (SemanticChangeSet cs : EvaluationUtil.getPropagationInterModelChanges(difference.getSymmetric())) {
					propagation.put(cs.getEditRName(), propagation.get(cs.getEditRName()) + 1);
				}				
			}
		}

		// Sort by operation name
		List<String> consistencyNames = new ArrayList<String>(consistency.keySet());
		Collections.sort(consistencyNames);
		List<String> couplingNames = new ArrayList<String>(coupling.keySet());
		Collections.sort(couplingNames);
		List<String> decouplingNames = new ArrayList<String>(decoupling.keySet());
		Collections.sort(decouplingNames);
		List<String> propagationNames = new ArrayList<String>(propagation.keySet());
		Collections.sort(propagationNames);
		
		// Add entries
		for (String key : consistencyNames) {
			ConcreteTestCaseEntry entry = new ConcreteTestCaseEntry(getEvaluation().getHeader("consistency")
					.getSubHeader(key), consistency.get(key));
			entries.add(entry);
		}
		for (String key : couplingNames) {
			ConcreteTestCaseEntry entry = new ConcreteTestCaseEntry(getEvaluation().getHeader("coupling")
					.getSubHeader(key), coupling.get(key));
			entries.add(entry);
		}
		for (String key : decouplingNames) {
			ConcreteTestCaseEntry entry = new ConcreteTestCaseEntry(getEvaluation().getHeader("decoupling")
					.getSubHeader(key), decoupling.get(key));
			entries.add(entry);
		}
		for (String key : propagationNames) {
			ConcreteTestCaseEntry entry = new ConcreteTestCaseEntry(getEvaluation().getHeader("propagation")
					.getSubHeader(key), propagation.get(key));
			entries.add(entry);
		}

		return entries;
	}

	@Override
	public String getKey() {
		return "DistributionEvaluation";
	}

}
