package saft.transformations.manual.validation;

import java.io.File;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.emf.henshin.trace.TracePackage;
import org.eclipse.equinox.app.IApplication;
import org.eclipse.equinox.app.IApplicationContext;
import org.osgi.framework.BundleContext;
import org.sidiff.common.logging.LogEvent;
import org.sidiff.common.logging.LogUtil;
import org.sidiff.difference.lifting.recognitionrulesorter.IRecognitionRuleSorter;
import org.sidiff.difference.lifting.recognitionrulesorter.util.RecognitionRuleSorterUtil;
import org.sidiff.difference.matcher.IMatcher;
import org.sidiff.difference.matcher.IncrementalMatcher;
import org.sidiff.difference.matcher.util.MatcherUtil;
import org.sidiff.difference.rulebase.extension.IRuleBase;
import org.sidiff.difference.rulebase.util.RuleBaseUtil;
import org.sidiff.difference.technical.ITechnicalDifferenceBuilder;
import org.sidiff.difference.technical.IncrementalTechnicalDifferenceBuilder;
import org.sidiff.difference.technical.util.TechnicalDifferenceBuilderUtil;

import FaultTree.FaultTreePackage;
import SA.SAPackage;

/**
 * Base class for running through the evolution scenarios and starting the
 * difference calculation with SiLift.
 * 
 * @author kehrer
 */
public abstract class AbstractDiffCreator implements IApplication {

	private static final String[] SCNEARIOS = new String[] { "00_02", "02_03", "03_04", "04_04b", "04b_07", "07_08",
			"08_09", "09_10", "10_11", "11_13", "13_14" };

	private static final String saPath = "de.stuttgart.informatik.PPU.SAmodels/model";
	private static final String ft1Path = "de.stuttgart.informatik.PPU.FT1models/model";
	private static final String ft2Path = "de.stuttgart.informatik.PPU.FT2models/model";
	private static final String outputPath = "saft.transformations.manual.validation.results/results";

	protected String outputFolder;

	@SuppressWarnings("unused")
	private static BundleContext context = null;

	public static void start(BundleContext context) {
		AbstractDiffCreator.context = context;

		// Init meta-models
		SAPackage.eINSTANCE.eClass();
		FaultTreePackage.eINSTANCE.eClass();
		TracePackage.eINSTANCE.eClass();
	}

	public static void stop(BundleContext context) {
		AbstractDiffCreator.context = null;
	}

	@Override
	public Object start(IApplicationContext context) throws Exception {

		IPath wsPath = ResourcesPlugin.getWorkspace().getRoot().getLocation();
		outputFolder = wsPath.append(outputPath).toOSString();

		LogUtil.log(LogEvent.NOTICE, "SA Path: " + saPath);
		LogUtil.log(LogEvent.NOTICE, "SA Path: " + ft1Path);
		LogUtil.log(LogEvent.NOTICE, "SA Path: " + ft2Path);
		LogUtil.log(LogEvent.NOTICE, "Output folder: " + outputFolder);

		// Collect scenarios
		List<Scenario> scenarios = new ArrayList<Scenario>();
		for (String sc : SCNEARIOS) {
			Scenario scenario = new Scenario();
			scenario.name = sc;
			String[] ids = sc.split("_");

			scenario.saOrigin = saPath + File.separator + "scenario_" + ids[0] + ".sa";
			scenario.saChanged = saPath + File.separator + "scenario_" + ids[1] + ".sa";
			scenario.ft1Origin = ft1Path + File.separator + "scenario_" + ids[0] + ".faulttree";
			scenario.ft1Changed = ft1Path + File.separator + "scenario_" + ids[1] + ".faulttree";
			scenario.ft2Origin = ft2Path + File.separator + "scenario_" + ids[0] + ".faulttree";
			scenario.ft2Changed = ft2Path + File.separator + "scenario_" + ids[1] + ".faulttree";

			scenario.init();
			scenarios.add(scenario);
		}

		for (Scenario scenario : scenarios) {
			LogUtil.log(LogEvent.NOTICE, scenario);

			// start difference calculation
			diff(scenario);
		}

		return IApplication.EXIT_OK;
	}

	protected abstract void diff(Scenario scenario);

	protected IMatcher getSA_Matcher(Scenario scenario) {
		return MatcherUtil.getMatcherByKey("SaMatcher", scenario.getResSaOrigin(), scenario.getResSaChanged());
	}

	protected IMatcher getFT_Matcher(Scenario scenario) {
		return MatcherUtil.getMatcherByKey("FtMatcher", scenario.getResFt1Origin(), scenario.getResFt1Changed());
	}

	protected IMatcher getTrace_Matcher(Scenario scenario) {
		return MatcherUtil.getMatcherByKey("TraceMatcher", scenario.getResFt1Origin(), scenario.getResFt1Changed());
	}

	protected IMatcher getIncrementalMatcher(Scenario scenario) {
		List<IMatcher> matchers = new ArrayList<IMatcher>();
		matchers.add(getSA_Matcher(scenario));
		matchers.add(getFT_Matcher(scenario));
		matchers.add(getTrace_Matcher(scenario));

		return new IncrementalMatcher(matchers);
	}

	protected ITechnicalDifferenceBuilder getSA_Builder(Scenario scenario) {
		return TechnicalDifferenceBuilderUtil.getDefaultTechnicalDifferenceBuilder(SAPackage.eNS_URI);
	}

	protected ITechnicalDifferenceBuilder getFT_Builder(Scenario scenario) {
		return TechnicalDifferenceBuilderUtil.getDefaultTechnicalDifferenceBuilder(FaultTreePackage.eNS_URI);
	}

	protected ITechnicalDifferenceBuilder getTrace_Builder(Scenario scenario) {
		return TechnicalDifferenceBuilderUtil.getDefaultTechnicalDifferenceBuilder(TracePackage.eNS_URI);
	}

	protected ITechnicalDifferenceBuilder getIncrementalBuilder(Scenario scenario) {
		List<ITechnicalDifferenceBuilder> builders = new ArrayList<ITechnicalDifferenceBuilder>();
		builders.add(getSA_Builder(scenario));
		builders.add(getFT_Builder(scenario));
		builders.add(getTrace_Builder(scenario));

		return new IncrementalTechnicalDifferenceBuilder(builders);
	}

	protected Set<IRuleBase> get_SA_RuleBases() {
		Set<String> documentTypes = new HashSet<String>();
		documentTypes.add(SAPackage.eNS_URI);
		documentTypes.add(TracePackage.eNS_URI);

		Set<IRuleBase> rbs = RuleBaseUtil.getAvailableRulebases(documentTypes);
		for (Iterator<IRuleBase> iterator = rbs.iterator(); iterator.hasNext();) {
			IRuleBase rb = iterator.next();
			if (rb.getName().contains("Atomic")) {
				iterator.remove();
			}
		}

		return rbs;
	}

	protected Set<IRuleBase> get_FT_RuleBases() {
		Set<String> documentTypes = new HashSet<String>();
		documentTypes.add(FaultTreePackage.eNS_URI);
		documentTypes.add(TracePackage.eNS_URI);

		Set<IRuleBase> rbs = RuleBaseUtil.getAvailableRulebases(documentTypes);
		for (Iterator<IRuleBase> iterator = rbs.iterator(); iterator.hasNext();) {
			IRuleBase rb = iterator.next();
			if (rb.getName().contains("Atomic")) {
				iterator.remove();
			}
		}

		return rbs;
	}

	protected Set<IRuleBase> get_ALL_RuleBases() {
		Set<String> documentTypes = new HashSet<String>();
		documentTypes.add(SAPackage.eNS_URI);
		documentTypes.add(FaultTreePackage.eNS_URI);
		documentTypes.add(TracePackage.eNS_URI);

		Set<IRuleBase> rbs = RuleBaseUtil.getAvailableRulebases(documentTypes);
		for (Iterator<IRuleBase> iterator = rbs.iterator(); iterator.hasNext();) {
			IRuleBase rb = iterator.next();
			if (rb.getName().contains("Atomic")) {
				iterator.remove();
			}
		}

		return rbs;
	}

	protected IRecognitionRuleSorter getSAFTRecognitionRuleSorter() {
		Set<IRecognitionRuleSorter> sorters = RecognitionRuleSorterUtil
				.getAvailableRecognitionRuleSorters(SAPackage.eNS_URI);
		for (IRecognitionRuleSorter iRecognitionRuleSorter : sorters) {
			if (iRecognitionRuleSorter.getKey().equals("SAFT_RRSorter")) {
				return iRecognitionRuleSorter;
			}
		}

		return null;
	}

	@Override
	public void stop() {

	}
}
