/**
 * @file TestDist.cpp
 * @date Aug 8, 2008
 * @author Ronald Kluth
 *
 * @brief Tests for ODEMx class Dist
 */

#include "TestRandom.h"
#include "../TestBase/TestBase.h"

/// @cond DOXYGEN_SKIP
SUITE( Random )
{
/// @endcond

	/**
	 * @struct DistFixture
	 * @brief Helper struct providing set-up/tear-down of Dist tests
	 *
	 * @copydetails EventFixture
	 */
	struct DistFixture
	{
		SuiteBase::SimulationTest sim;
		DistTest dist;

		DistFixture()
		:	sim( "DistTestSim" ),
			dist( sim, "DistTest" )
			{}
	};

	/**
	 * @test odemx::Dist construction
	 *
	 * Expected function call effects:
	 * @li dist context (simulation) is set correctly
	 * @li label is set correctly
	 */
	TEST_FIXTURE( DistFixture, ConstructionDestruction )
	{
		data::Label l = "DistTestConstruction";
		DistTest dist( sim, l );
		CHECK_EQUAL( l, dist.getLabel() );
	}

	/**
	 * @test odemx::Dist::setSeed(int)
	 *
	 * Expected function call effects:
	 * @li allowed values are set as is
	 * @li a negative value is turned into a positive value
	 * @li a too large value is decreased by \c odemx::zyqmodulo
	 * @li a value of \c 0 will cause seed to be set to a default value
	 */
	TEST_FIXTURE( DistFixture, SetSeed )
	{
		unsigned long seed = 125;
		dist.setSeed( seed );
		CHECK_EQUAL( seed, dist.getSeed() );

		seed = 125;
		long negativeSeed = 0 - seed;
		dist.setSeed( negativeSeed );
		CHECK_EQUAL( seed, dist.getSeed() );

		seed = zyqmodulo + 25;
		dist.setSeed( seed );
		CHECK_EQUAL( seed - ( (unsigned long)( seed/zyqmodulo) )*zyqmodulo, dist.getSeed() );

		dist.setSeed( 0 );
		CHECK_EQUAL( (unsigned long)( zyqmodulo / 2 ), dist.getSeed() );
	}

	/**
	 * @test odemx::Dist::getSample()
	 *
	 * Expected function call effects:
	 * @li all sample values fall within the interval [0,1]
	 */
	TEST_FIXTURE( DistFixture, GetSample )
	{
		for( int i = 0; i < 10000; ++i )
		{
			CHECK( 0 <= dist.getSample() );
			CHECK( 1 >= dist.getSample() );
		}
	}

/// @cond DOXYGEN_SKIP
}
/// @endcond
