//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file CellVariablesContainer.h

 \author Sascha Qualitz

 \date created at 2010/01/05

 \brief Declaration of CellVariablesContainer

 \since 3.0
 */

#ifndef ODEMX_CELLVARIABLESCONTAINER_INCLUDED
#define ODEMX_CELLVARIABLESCONTAINER_INCLUDED

#define SETTYPE(arraytype) arraytype*


namespace odemx {
	namespace base {

		/** \class CellVariablesContainer

			\ingroup base

			\brief abstract class for implementing memory handling of a cellular automaton

			\sa Cell, CellMonitor

			\since 3.0
		 */
		//template <class T = double>
		class CellVariablesContainer {
		public:
			/// construction
			CellVariablesContainer(unsigned row, unsigned column, unsigned variablesCount);
			/// destruction
			virtual ~CellVariablesContainer();

			/** \brief set output value of one variable
				\param cellIndex
					index of the cell to set \p value
				\param variableIndex
					index of the variable to set \p value
				\param value
					the value to set
			 */
			virtual void setOutputValue(unsigned cellIndex, unsigned variableIndex, int value);
			//virtual void setOutputValue(unsigned cellIndex, unsigned variableIndex, T value);

			/** \brief set output value of one variable
				\param cellIndex
					index of the cell to set \p value
				\param value
					the value to set

				\note variableIndex is set to 0
			 */
			//virtual void setValue(unsigned cellRow, unsigned cellColumn, int value);
			virtual void setOutputValue(unsigned cellIndex, int value);
			//virtual void setOutputValue(unsigned cellIndex, T value);

			/** \brief set state value of one variable
				\param cellIndex
					index of the cell to set \p value
				\param variableIndex
					index of the variable to set \p value
				\param value
					the value to set
			 */
			virtual void setStateValue(unsigned cellIndex, unsigned variableIndex, int value);
			//virtual void setStateValue(unsigned cellIndex, unsigned variableIndex, T value);

			/** \brief set state value of one variable
				\param cellIndex
					index of the cell to set \p value
				\param value
					the value to set

				\note variableIndex is set to 0
			 */
			//virtual void setValue(unsigned cellRow, unsigned cellColumn, int value);
			virtual void setStateValue(unsigned cellIndex, int value);
			//virtual void setStateValue(unsigned cellIndex, T value);

			/** \brief set input value of one variable
				\param cellIndex
					index of the cell to set \p value
				\param variableIndex
					index of the variable to set \p value
				\param value
					the value to set
			 */
			virtual void setInputValue(unsigned cellIndex, unsigned variableIndex, int value);
			//virtual void setInputValue(unsigned cellIndex, unsigned variableIndex, T value);

			/** \brief set input value of one variable
				\param cellIndex
					index of the cell to set \p value
				\param value
					the value to set

				\note variableIndex is set to 0
			 */
			//virtual void setValue(unsigned cellRow, unsigned cellColumn, int value);
			virtual void setInputValue(unsigned cellIndex, int value);
			//virtual void setInputValue(unsigned cellIndex, T value);

			/** \brief get output value of one variable
				\param cellIndex
					index of the cell to get \p value
				\param variableIndex
					index of the variable to g	et \p value
			 */
			virtual int getOutputValue(unsigned cellIndex, unsigned variableIndex);
			//virtual T getOutputValue(unsigned cellIndex, unsigned variableIndex);

			/** \brief get output value of one variable
				\param cellIndex
					index of the cell to get \p value

				\note variableIndex is set to 0
			 */
			virtual int getOutputValue(unsigned cellIndex);
			//virtual T getOutputValue(unsigned cellIndex);

			/** \brief get state value of one variable
				\param cellIndex
					index of the cell to get \p value
				\param variableIndex
					index of the variable to g	et \p value
			 */
			virtual int getStateValue(unsigned cellIndex, unsigned variableIndex);
			//virtual T getStateValue(unsigned cellIndex, unsigned variableIndex);

			/** \brief get state value of one variable
				\param cellIndex
					index of the cell to get \p value

				\note variableIndex is set to 0
			 */
			virtual int getStateValue(unsigned cellIndex);
			//virtual T getStateValue(unsigned cellIndex);

			/** \brief get input value of one variable
				\param cellIndex
					index of the cell to get \p value
				\param variableIndex
					index of the variable to g	et \p value
			 */
			virtual int getInputValue(unsigned cellIndex, unsigned variableIndex);
			//virtual T getInputValue(unsigned cellIndex, unsigned variableIndex);

			/** \brief get input value of one variable
				\param cellIndex
					index of the cell to get \p value

				\note variableIndex is set to 0
			 */
			virtual int getInputValue(unsigned cellIndex);
			//virtual T getInputValue(unsigned cellIndex);

			/** \brief set value of one variable
				\param cellRow
					row index of the cell to set \p value
				\param cellColumn
					column index of the cell to set \p value
				\param variableIndex
					index of the variable to set \p value
				\param value
					the value to set
			 */
			//virtual void setValue(unsigned cellRow, unsigned cellColumn, unsigned variableIndex, int value);

			/** \brief set value of one variable
				\param cellRow
					row index of the cell to set \p value
				\param cellColumn
					column index of the cell to set \p value
				\param value
					the value to set

				\note variableIndex is set to 0
			 */
			//virtual void setValue(unsigned cellRow, unsigned cellColumn, int value);

			/** \brief get value of one variable
				\param cellRow
					row index of the cell to get \p value
				\param cellColumn
					column index of the cell to get \p value
				\param variableIndex
					index of the variable to get \p value
			 */
			//virtual int getValue(unsigned cellRow, unsigned cellColumn, unsigned variableIndex);

			/** \brief get value of one variable
				\param cellRow
					row index of the cell to get \p value
				\param cellColumn
					column index of the cell to get \p value

				\note variableIndex is set to 0
			 */
			//virtual int getValue(unsigned cellRow, unsigned cellColumn);

			/** \brief returns the number of rows that belongs to the cellular automaton
			 */
			unsigned getNumberOfRows();

			/** \brief returns the number of columns that belongs to the cellular automaton
			 */
			unsigned getNumberOfColumns();

			/** \brief initialize arrays
			 */
			void initializeArrays();

			/** \brief swaps the temp_values to the values array, deletes the values array and creates a new temp_values array
			 */
			void copyValues();

			/**
			 * temp function for printing values
			 */
			//T* getOutputValues();
			int* getOutputValues();

		private:

			/** \brief Array of cell values
			 */
			//T* values;
			/** \brief Array of temporarily cell values
			 */
			//T* temp_values;

			/** \brief number of rows that belongs to the cellular automaton
			 */
			unsigned row;

			/** \brief number of columns that belongs to the cellular automaton
			 */
			unsigned column;

			/** \brief number of variables that belongs to a cell of the cellular automaton
			 */
			unsigned variablesCount;

			/** \brief Array of all base indices for the cells of the cellular automaton
			*/
			int* baseIndices;

			int* input_values;
			int* state_values;
			int* output_values;

			//T* input_values;
			//T* state_values;
			//T* output_values;

		};

	}
}
#endif


//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file CellVariablesContainer.h

 \author Sascha Qualitz

 \date created at 2010/01/05

 \brief Implementation of class CoordinateContainer

 \sa VariableContainer.h

 \since 3.0
 */

//#include <odemx/base/cellular_automaton/CellVariablesContainer.h>

/*
using namespace odemx::base;
using namespace std;

//template <typename T>
CellVariablesContainer<T>::CellVariablesContainer(unsigned row, unsigned column, unsigned variablesCount)
//CellVariablesContainer::CellVariablesContainer(unsigned row, unsigned column, unsigned variablesCount)
: variablesCount(variablesCount),
  row(row),
  column(column)
{
	//this->values = new T[variablesCount];
	//this->temp_values = new T[variablesCount];
	values = new int *[row*column];
	*values = new int [variablesCount];
	temp_values = new int *[row*column];
	*temp_values = new int [variablesCount];

	input_values = new T[row*column*variablesCount];
	state_values = new T[row*column*variablesCount];
	output_values = new T[row*column*variablesCount];

	//this->baseIndices = new int[variablesCount];
	//std::cout << "CellVariablesContainer::CellVariablesContainer" << std::endl;
}

template <typename T>
CellVariablesContainer<T>::~CellVariablesContainer() {
//CellVariablesContainer::~CellVariablesContainer() {
	//nothing todo
}

void CellVariablesContainer::setValue(unsigned cellRow, unsigned cellColumn, unsigned variableIndex, int value) {
	this->input_values[cellRow*this->column*this->variablesCount+cellColumn*this->variablesCount+variableIndex] = value;
}

template <typename T>
void CellVariablesContainer<T>::setOutputValue(unsigned cellIndex, unsigned variableIndex, T value) {
//void CellVariablesContainer::setOutputValue(unsigned cellIndex, unsigned variableIndex, int value) {
	this->output_values[cellIndex*this->variablesCount+variableIndex] = value;
}

template <typename T>
void CellVariablesContainer<T>::setOutputValue(unsigned cellIndex, T value) {
//void CellVariablesContainer<T>::setOutputValue(unsigned cellIndex, int value) {
	this->output_values[cellIndex] = value;
}

template <typename T>
T CellVariablesContainer<T>::getOutputValue(unsigned cellIndex, unsigned variableIndex) {
//int CellVariablesContainer<T>::getOutputValue(unsigned cellIndex, unsigned variableIndex) {
	return this->output_values[cellIndex*this->variablesCount+variableIndex];
}

template <typename T>
T CellVariablesContainer<T>::getOutputValue(unsigned cellIndex) {
//int CellVariablesContainer<T>::getOutputValue(unsigned cellIndex) {
	return this->output_values[cellIndex];
}

template <typename T>
void CellVariablesContainer<T>::setStateValue(unsigned cellIndex, unsigned variableIndex, T value) {
	this->state_values[cellIndex*this->variablesCount+variableIndex] = value;
}

template <typename T>
void CellVariablesContainer<T>::setStateValue(unsigned cellIndex, T value) {
	this->state_values[cellIndex] = value;
}

template <typename T>
T CellVariablesContainer<T>::getStateValue(unsigned cellIndex, unsigned variableIndex) {
	return this->state_values[cellIndex*this->variablesCount+variableIndex];
}

template <typename T>
T CellVariablesContainer<T>::getStateValue(unsigned cellIndex) {
	return this->state_values[cellIndex];
}

template <typename T>
void CellVariablesContainer<T>::setInputValue(unsigned cellIndex, unsigned variableIndex, T value) {
	this->input_values[cellIndex*this->variablesCount+variableIndex] = value;
}

template <typename T>
void CellVariablesContainer<T>::setInputValue(unsigned cellIndex, T value) {
	this->input_values[cellIndex] = value;
}

template <typename T>
T CellVariablesContainer<T>::getInputValue(unsigned cellIndex, unsigned variableIndex) {
	return this->input_values[cellIndex*this->variablesCount+variableIndex];
}

template <typename T>
T CellVariablesContainer<T>::getInputValue(unsigned cellIndex) {
	return this->input_values[cellIndex];
}

void CellVariablesContainer::setValue(unsigned cellRow, unsigned cellColumn, unsigned variableIndex, int value) {
	this->input_values[cellRow*this->column*this->variablesCount+cellColumn*this->variablesCount+variableIndex] = value;
}

void CellVariablesContainer::setValue(unsigned cellRow, unsigned cellColumn, int value) {
	this->input_values[cellRow*this->column+cellColumn] = value;
}

int CellVariablesContainer::getValue(unsigned cellRow, unsigned cellColumn, unsigned variableIndex) {
	return this->output_values[cellRow*this->column*this->variablesCount+cellColumn*this->variablesCount+variableIndex];
}

int CellVariablesContainer::getValue(unsigned cellRow, unsigned cellColumn) {
	return this->output_values[cellRow*this->column+cellColumn];
}

template <typename T>
unsigned CellVariablesContainer<T>::getNumberOfRows() {
	return this->row;
}

template <typename T>
unsigned CellVariablesContainer<T>::getNumberOfColumns() {
	return this->column;
}

template <typename T>
void CellVariablesContainer<T>::initializeArrays() {
	for (int i = 0; i < row*column*variablesCount; ++i) {
		this->input_values[i] = this->state_values[i];
	}
}

template <typename T>
void CellVariablesContainer<T>::copyValues() {
	for (int i = 0; i < row*column*variablesCount; ++i) {
		this->output_values[i] = this->state_values[i];
		this->input_values[i] = this->output_values[i];
	}
}

//----------------------------------------------------------- temp function for printing values

template <typename T>
T* CellVariablesContainer<T>::getOutputValues() {
	return this->output_values;
}

*/
