//
// SortedTableModel.cpp
//
// $Id: //poco/Main/WebWidgets/src/SortedTableModel.cpp#3 $
//
// Library: WebWidgets
// Package: Views
// Module:  SortedTableModel
//
// Copyright (c) 2007, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#include "Poco/WebWidgets/SortedTableModel.h"
#include "Poco/WebWidgets/Table.h"
#include <set>


namespace Poco {
namespace WebWidgets {


SortedTableModel::SortedTableModel(const Table* pTable, std::size_t col, bool sortAscending):
	TableModel(pTable->getModel().getColumnCount()),
	_unsorted(const_cast<TableModel&>(pTable->getModel())),
	_pTable(pTable),
	_sortCol(col),
	_sortAscending(sortAscending),
	_mapping(pTable->getModel().getRowCount())
{
	forceResort(col, _sortAscending);
}


SortedTableModel::~SortedTableModel()
{
}


const Poco::Any& SortedTableModel::getValue(std::size_t row, std::size_t col) const
{
	return _unsorted.getValue(mapping(row), col);
}


std::size_t SortedTableModel::getRowCount() const
{
	return _unsorted.getRowCount();
}


void SortedTableModel::setValue(const Poco::Any& val, std::size_t row, std::size_t col)
{
	_unsorted.setValue(val, mapping(row), col);
}


void SortedTableModel::deleteRow(std::size_t row)
{
	std::size_t mappedRow = mapping(row);
	_unsorted.deleteRow(mappedRow);
	_mapping.erase(_mapping.begin()+row);
}

	
void SortedTableModel::clear()
{
	_unsorted.clear();
	_mapping.clear();
}


typedef std::pair<const Poco::Any*, std::size_t> Val;
struct less
{
	Formatter& fmt;
	less(Formatter& f):fmt(f){}
	bool operator () (const Val& first, const Val& second) const 
	{ 
		return fmt.lowerThan(*first.first, *second.first);
	}
};


void SortedTableModel::sort(std::size_t col, bool sortAscending)
{
	if (_sortCol == col)
	{
		// we already have a sort on this oclumn
		_sortAscending = sortAscending;
		return;
	}
	
	forceResort(col, sortAscending);
}


void SortedTableModel::forceResort(std::size_t col, bool sortAscending)
{
	_sortCol = col;
	_sortAscending = sortAscending;
	//build up a table of pairs containg <val, row>
	// create a set of of these pairs to e
	Formatter::Ptr pF = _pTable->getColumns()[col]->getCell()->getFormatter();
	less ls(*pF);
	std::multiset<Val, less> tbl(ls); 	
	
	for (std::size_t row = 0; row < _unsorted.getRowCount(); ++row)
	{
		tbl.insert(std::make_pair(&_unsorted.getValue(row, col), row));
	}
	
	_mapping.clear();
	
	std::multiset<Val, less>::const_iterator it = tbl.begin();
	for (; it != tbl.end(); ++it)
		_mapping.push_back(it->second);
}


TableModel::Ptr SortedTableModel::clone() const
{
	SortedTableModel::Ptr pModel(new SortedTableModel(_pTable, _sortCol, _sortAscending));
	return pModel;
}


} } // namespace Poco::WebWidgets
