//
// RequestHandler.cpp
//
// $Id: //poco/Main/WebWidgets/src/RequestHandler.cpp#4 $
//
// Library: WebWidgets
// Package: Core
// Module:  RequestHandler
//
// Definition of the RequestHandler class.
//
// Copyright (c) 2008, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#include "Poco/WebWidgets/RequestHandler.h"
#include "Poco/WebWidgets/WebApplication.h"
#include "Poco/WebWidgets/Page.h"
#include "Poco/WebWidgets/RenderContext.h"
#include "Poco/WebWidgets/RequestProcessor.h"
#include "Poco/WebWidgets/SubmitButtonCell.h"
#include "Poco/WebWidgets/WebWidgetsException.h"
#include "Poco/Net/HTTPServerRequest.h"
#include "Poco/Net/HTTPServerResponse.h"
#include "Poco/NumberParser.h"
#include "Poco/URI.h"
#include "Poco/ThreadLocal.h"
#include <sstream>


using Poco::URI;


namespace Poco {
namespace WebWidgets {


const std::string RequestHandler::KEY_ID("id");
const std::string RequestHandler::KEY_EVID("evId");


RequestHandler::RequestHandler():
	_pApp()
{
}


RequestHandler::RequestHandler(Poco::SharedPtr<WebApplication> app):
	_pApp(app)
{
}


RequestHandler::~RequestHandler()
{
}


void RequestHandler::handleRequest(Poco::Net::HTTPServerRequest& request, Poco::Net::HTTPServerResponse& response)
{
	_pApp->attachToThread(request);
	Poco::Net::NameValueCollection args;
	parseRequest(request, args);
	
	if (args.empty())
	{
		Poco::Net::HTMLForm form(request, request.stream());
		if (!form.empty())
		{
			try
			{
				handleForm(form);
				Poco::Net::NameValueCollection::ConstIterator it = form.find(Form::FORM_ID);
				if (it != form.end())
				{
					_pApp->notifySubmitButton(Poco::NumberParser::parse(it->second));
				}
				response.send();
			}
			catch(WebWidgetsException& e)
			{
				Poco::Net::HTTPResponse::HTTPStatus code = Poco::Net::HTTPResponse::HTTP_BAD_REQUEST;
				if (e.code() > code && e.code() <= Poco::Net::HTTPResponse::HTTP_EXPECTATION_FAILED)
					code = (Poco::Net::HTTPResponse::HTTPStatus)e.code();
				response.setStatusAndReason(code, e.displayText());
				response.send();
			}
			catch(Poco::Exception& e)
			{
				response.setStatusAndReason(Poco::Net::HTTPResponse::HTTP_INTERNAL_SERVER_ERROR, e.displayText());
				response.send();
			}
			catch(std::exception& e)
			{
				response.setStatusAndReason(Poco::Net::HTTPResponse::HTTP_INTERNAL_SERVER_ERROR, e.what());
				response.send();
			}
			catch(...)
			{
				response.setStatusAndReason(Poco::Net::HTTPResponse::HTTP_INTERNAL_SERVER_ERROR, "Unknown exception");
				response.send();
			}
			
		}
		else
			handlePageRequest(request, response);
	}
	else
	{
		Poco::Net::HTMLForm form(request, request.stream());
		// we provide the form data transparently to the ajax request handler
		// by simply adding it to the args already received via AJAX URI
		Poco::Net::NameValueCollection::ConstIterator it = form.begin();
		for (;it != form.end(); ++it)
		{
			const std::string& key = it->first;
			const std::string& value = it->second;
			args.add(key, value);
		}
		handleAjaxRequest(request, response, args);
	}
}


void RequestHandler::handlePageRequest(Poco::Net::HTTPServerRequest& request, Poco::Net::HTTPServerResponse& response)
{
	response.setChunkedTransferEncoding(true);
	response.setContentType("text/html");
	std::ostream& str = response.send();
	RenderContext renderContext(*_pApp->getLookAndFeel(), *_pApp);
	Page::Ptr pPage = _pApp->getCurrentPage();
	if (pPage)
	{
		pPage->renderHead(renderContext, str);
		pPage->renderBody(renderContext, str);
	}
}


void RequestHandler::handleAjaxRequest(Poco::Net::HTTPServerRequest& request, Poco::Net::HTTPServerResponse& response, const Poco::Net::NameValueCollection& args)
{
	Poco::Net::NameValueCollection::ConstIterator it = args.find(KEY_ID);
	if (it == args.end())
	{
		response.setStatusAndReason(Poco::Net::HTTPResponse::HTTP_BAD_REQUEST,"missing id");
		response.send();
		return;
	}

	const std::string id = it->second;

	it = args.begin();
	RequestProcessor* pProc = _pApp->getAjaxProcessor(id);
	//poco_assert_dbg (pProc);
	if (!pProc)
	{
		response.setStatusAndReason(Poco::Net::HTTPResponse::HTTP_INTERNAL_SERVER_ERROR, "no requestprocessor found");
		response.send();
		return;
	}
	
	try
	{
		pProc->handleAjaxRequest(args, response);
	}
	catch(WebWidgetsException& e)
	{
		Poco::Net::HTTPResponse::HTTPStatus code = Poco::Net::HTTPResponse::HTTP_BAD_REQUEST;
		if (e.code() > code && e.code() <= Poco::Net::HTTPResponse::HTTP_EXPECTATION_FAILED)
			code = (Poco::Net::HTTPResponse::HTTPStatus)e.code();
		response.setStatusAndReason(code, e.displayText());
		response.send();
	}
	catch(Poco::Exception& e)
	{
		response.setStatusAndReason(Poco::Net::HTTPResponse::HTTP_INTERNAL_SERVER_ERROR, e.displayText());
		response.send();
	}
	catch(std::exception& e)
	{
		response.setStatusAndReason(Poco::Net::HTTPResponse::HTTP_INTERNAL_SERVER_ERROR, e.what());
		response.send();
	}
	catch(...)
	{
		response.setStatusAndReason(Poco::Net::HTTPResponse::HTTP_INTERNAL_SERVER_ERROR, "Unknown exception");
		response.send();
	}
}


void RequestHandler::handleForm(const Poco::Net::HTMLForm& form)
{
	_pApp->handleForm(form);
}


void RequestHandler::parseRequest(const Poco::Net::HTTPServerRequest& request, Poco::Net::NameValueCollection& nvc)
{
	URI uri(request.getURI());
	std::string path = uri.getPath();
	std::string args;
	std::string::size_type pos = path.find(';');
	if (pos != std::string::npos)
	{
		parseRequest(path.substr(pos + 1), nvc);
	}
}


void RequestHandler::parseRequest(const std::string& path, Poco::Net::NameValueCollection& nvc)
{
	static const int eof = std::char_traits<char>::eof();

	std::istringstream istr(path);
	int ch = istr.get();
	while (ch != eof)
	{
		std::string name;
		std::string value;
		while (ch != eof && ch != '=' && ch != '&')
		{
			if (ch == '+') ch = ' ';
			name += (char) ch;
			ch = istr.get();
		}
		if (ch == '=')
		{
			ch = istr.get();
			while (ch != eof && ch != '&')
			{
				if (ch == '+') ch = ' ';
				value += (char) ch;
				ch = istr.get();
			}
		}
		std::string decodedName;
		std::string decodedValue;
		URI::decode(name, decodedName);
		URI::decode(value, decodedValue);
		nvc.add(decodedName, decodedValue);
		if (ch == '&') ch = istr.get();
	}
}

} } // namespace Poco::WebWidgets
