//
// Template.h
//
// $Id: //poco/Main/WebWidgets/include/Poco/WebWidgets/Template.h#2 $
//
// Library: WebWidgets
// Package: Templates
// Module:  Template
//
// Definition of the Template class.
//
// Copyright (c) 2007, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef WebWidgets_Template_INCLUDED
#define WebWidgets_Template_INCLUDED


#include "Poco/WebWidgets/View.h"
#include "Poco/WebWidgets/Getter.h"
#include "Poco/FIFOEvent.h"
#include "Poco/Any.h"
#include <ostream>


namespace Poco {
namespace WebWidgets {


class RenderContext;


class WebWidgets_API Template: public View
	/// A JavaScript template class: Defines a template string which contains wildcards of the form %0, %1 %2 ...
	/// You can bind values to the wildcards. Values must be of type:
	///  - string, int, double, float, char, bool,  Poco::WebWidgets::Renderable, DateTime
	/// properties inherited by View (like width, height) are ignored
{
public:
	typedef Poco::AutoPtr<Template> Ptr;
	
	Template(const std::string& templateStr);
		/// Creates a template string
	
	Template(const std::string& templateStr, const std::vector<Poco::Any>& values);
		/// Creates a Template.
	
	template<typename T>
	void setValue(std::size_t num, const T& val)
		/// Sets a value
	{
		bind(num, val);
	}
	
	template<typename T>
	void setValue(std::size_t num, Poco::AutoPtr<T> val)
	{
		bindPtr(num, val);
	}
	
	template<typename T>
	void setValue(std::size_t num, Poco::SharedPtr<T> val)
	{
		bindPtr(num, val);
	}
	
	template<typename T>
	const T& getValue(std::size_t num) const
	{
		return Poco::RefAnyCast<T>(_values[num]);
	}
	
		
	const std::string& templateStr() const;
		/// Returns the templatestr
		
	std::size_t maxWildCard() const;
		/// Returns the largest wildcard
		
	void parse(const RenderContext& ctx, std::ostream& out) const;
		/// The parse method will iterate over the template and write the expanded template to out
		
protected:
	Template(const std::string& templateStr, const std::type_info& type);
		/// Creates a Template.

	~Template();
		/// Destroys the Template.
	
	std::size_t detectMaxWildCard() const;
	
	void bind(std::size_t num, const std::vector<Poco::Any>& val);
	
	void bind(std::size_t num, int val);
	void bind(std::size_t num, float val);
	void bind(std::size_t num, double val);
	void bind(std::size_t num, char val);
	void bind(std::size_t num, bool val);
	void bind(std::size_t num, const std::string& val);
	void bind(std::size_t num, const char* pVal);
	void bind(std::size_t num, Renderable* pVal);
	void bind(std::size_t num, const Any& val);
	void bindPtr(std::size_t num, Poco::AutoPtr<Renderable> pRend);
	void bindPtr(std::size_t num, Poco::AutoPtr<Getter> pRend);
	void bindPtr(std::size_t num, Poco::SharedPtr<std::vector<Poco::Any> > pVal);
	void bindPtr(std::size_t num, Poco::SharedPtr<std::vector<float> > pVal);
	void bindPtr(std::size_t num, Poco::SharedPtr<std::vector<double> > pVal);
	void bindPtr(std::size_t num, Poco::SharedPtr<std::vector<char> > pVal);
	void bindPtr(std::size_t num, Poco::SharedPtr<std::vector<bool> > pVal);
	void bindPtr(std::size_t num, Poco::SharedPtr<std::vector<Renderable::Ptr> > pVal);
	void bindPtr(std::size_t num, Poco::SharedPtr<std::vector<std::string> > pVal);
	void bindPtr(std::size_t num, Poco::SharedPtr<std::vector<Getter::Ptr> > pVal);
	void bindPtr(std::size_t num, Poco::SharedPtr<int> pVal);
	void bindPtr(std::size_t num, Poco::SharedPtr<float> pVal);
	void bindPtr(std::size_t num, Poco::SharedPtr<double> pVal);
	void bindPtr(std::size_t num, Poco::SharedPtr<char> pVal);
	void bindPtr(std::size_t num, Poco::SharedPtr<bool> pVal);
	void bindPtr(std::size_t num, Poco::SharedPtr<std::string> pVal);
	void bindPtr(std::size_t num, Poco::SharedPtr<Poco::Any> pVal);
	void write(const RenderContext& ctx, std::ostream& out, const Any& val) const;
	
	template<typename T>
	void writeVector(const RenderContext& ctx, std::ostream& out, const std::vector<T>& val) const
	{
		for (std::size_t i = 0; i < val.size(); ++i)
		{
			if (i > 0)
				out << ",";
			write(ctx, out, val[i]);
		}
	}
	
private:
	std::string            _templateStr;
	std::size_t            _maxWildCard;
	std::vector<Poco::Any> _values;
};


inline const std::string& Template::templateStr() const
{
	return _templateStr;
}

		
inline std::size_t Template::maxWildCard() const
{
	return _maxWildCard;
}


} } // namespace Poco::WebWidgets


#endif // WebWidgets_Template_INCLUDED
