//
// Renderable.h
//
// $Id: //poco/Main/WebWidgets/include/Poco/WebWidgets/Renderable.h#2 $
//
// Library: WebWidgets
// Package: Core
// Module:  Renderable
//
// Definition of the Renderable class.
//
// Copyright (c) 2008, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef WebWidgets_Renderable_INCLUDED
#define WebWidgets_Renderable_INCLUDED


#include "Poco/WebWidgets/WebWidgets.h"
#include "Poco/WebWidgets/Renderer.h"
#include "Poco/RefCountedObject.h"
#include "Poco/AutoPtr.h"
#include "Poco/Mutex.h"
#include <typeinfo>


namespace Poco {
namespace WebWidgets {


class LookAndFeel;
class Renderer;
class RenderContext;


class WebWidgets_API Renderable: public Poco::RefCountedObject
	/// Renderable is the base class for all objects that will be visible to the 
	/// user in the browser, and thus need to produce HTML/CSS and/or JavaScript 
	/// code to be sent to the browser. Renderable objects just know about their 
	/// properties, but don’t know how to produce rendering code (HTML/CSS and/or 
	/// JavaScript). This is the task of Renderer classes. Separating Renderable 
	/// and Renderer classes allows for a dynamically changing look and feel of
	/// an application. 
	///
	/// Every instance of Renderable has a unique numeric ID, which is 
	/// automatically assigned by the framework.
	/// At runtime, Renderable objects find their Renderer objects using a 
	/// LookAndFeel object, which returns Renderer objects based on the 
	/// Renderable’s type information (std::type_info).
	///
	/// Rendering is done in a two-pass process. In the first pass, implemented 
	/// by renderHead(), code for the page header (usually CSS and JavaScript) 
	/// is emitted. In the second pass (implemented by renderBody()), HTML and
	/// JavaScript code for the HTML body is emitted.
{
public:
	typedef Poco::UInt64 ID;
	typedef Poco::AutoPtr<Renderable> Ptr;

	ID id() const;
		/// Returns the Renderable's unique (within the process) ID.

	Renderer& renderer(const LookAndFeel& lookAndFeel) const;
		/// Returns the renderer for the given LookAndFeel, based on
		/// the std::type_info set in the constructor.
		
	const std::type_info& type() const;
		/// Returns the type information for this object.
		
	virtual void renderHead(const RenderContext& context, std::ostream& str) const;
		/// Emits code for the page header to the given stream, using
		/// the LookAndFeel obtained from the RenderContext.
		///
		/// Can be overridden by subclasses; however, overriding implementations
		/// should call this implementation.
	
	virtual void renderBody(const RenderContext& context, std::ostream& str) const;
		/// Emits code for the page body to the given stream, using
		/// the LookAndFeel obtained from the RenderContext.
		///
		/// Can be overridden by subclasses; however, overriding implementations
		/// should call this implementation.
		
protected:
	Renderable(const std::type_info& type);
		/// Creates the Renderable object and assigns a unique ID.
		
	~Renderable();
		/// Destroys the Renderable object.

private:
	Renderable();
	Renderable(const Renderable&);
	Renderable& operator = (const Renderable&);
	
	ID _id;
	const std::type_info& _type;
	mutable const LookAndFeel* _pLookAndFeel;
	mutable Renderer::Ptr _pRenderer;

	static Poco::FastMutex _nextIdMutex;
	static ID _nextId;
};


//
// inlines
//
inline Renderable::ID Renderable::id() const
{
	return _id;
}


inline const std::type_info& Renderable::type() const
{
	return _type;
}


} } // namespace Poco::WebWidgets


#endif // WebWidgets_Renderable_INCLUDED
