//
// Page.h
//
// $Id: //poco/Main/WebWidgets/include/Poco/WebWidgets/Page.h#3 $
//
// Library: WebWidgets
// Package: Views
// Module:  Page
//
// Definition of the Page class.
//
// Copyright (c) 2008, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef WebWidgets_Page_INCLUDED
#define WebWidgets_Page_INCLUDED


#include "Poco/WebWidgets/ContainerView.h"
#include "Poco/WebWidgets/JavaScriptEvent.h"
#include "Poco/WebWidgets/ResourceManager.h"
#include "Poco/WebWidgets/RequestProcessor.h"
#include "Poco/WebWidgets/DynamicCodeLoader.h"
#include "Poco/Net/NameValueCollection.h"
#include "Poco/Net/HTTPServerResponse.h"
#include "Poco/BasicEvent.h"
#include <set>
#include <vector>


namespace Poco {
namespace WebWidgets {


class WebWidgets_API Page: public ContainerView, public RequestProcessor
	/// A Page displays a HTML Page
{
public:
	typedef Poco::AutoPtr<Page> Ptr;
	
	static const std::string EV_BEFORERENDER;
	static const std::string EV_AFTERRENDER;
	
	JavaScriptEvent<Page*> beforeRender; 
		/// event thrown before GUI rendering. 
		
	JavaScriptEvent<Page*> afterRender; 
		/// event thrown after GUI rendering. 
		
	Poco::BasicEvent<Page*> pageRequested;
		/// Thrown whenever the page is requested, before code is generated
		
	Poco::BasicEvent<Page*> afterPageRequested;
		/// Thrown whenever the page is requested, after code is generated
		
	Page();
		/// Creates an anonymous Page.
		
	Page(const std::string& name);
		/// Creates a Page with the given name.
	
	// View
	void setText(const std::string& text);
	
	std::string getText() const;
	
	ResourceManager& resourceManager();
		/// Returns the ResourceManager
		
	const ResourceManager& resourceManager() const;
		/// Returns the ResourceManager
		
	void handleAjaxRequest(const Poco::Net::NameValueCollection& args, Poco::Net::HTTPServerResponse& response);
		/// Handles a complete AJAX request submitted by the client.
		
	void handleForm(const std::string& field, const std::string& value);
		/// Dummy implementation
		
	bool serializeJSON(std::ostream& out, const std::string& name);
		/// Dummy implementation
		
	void addDynamicFunction(const std::string& jsCode);
		/// Adds a JavaScript function to the page. Static functions should be written to a JS file
		/// and included via the ResourceManager, only dynamic fucntions (ie. functions that are generated
		/// during run-time) should be included here
		
	const std::vector<std::string>& dynamicFunctions() const;
		/// Returns all dynamic functions
		
	void setPostRenderCode(const std::string& js);
		/// Sets Javascript code that should be executed after the page code was written. The difference to afterRender
		/// is that afterRender is a JS Event executed from within the rendering pipeline of the browser
		/// (ie. changes to the GUI might block the browser indefinitely) whereas
		/// post render code is executed immediately after the JS code that generates the GUI
		/// (ie. it is not necessarily rendered, no data loaded but thread-safe!)
		/// Old post render code gets deleted by this function.
		/// To preverse old code use appendPostRenderCode.
		
	const std::string& getPostRenderCode() const;
		/// Returns post render code. Can be an empty string
		
	void appendPostRenderCode(const std::string& js);
		/// Appends Javascript code that should be executed after the page code was written.
		
	void addDynamicCodeLoader(DynamicCodeLoader::Ptr pLoader);
	
	const std::set<DynamicCodeLoader::Ptr>& dynamicCodeLoaders() const;
	
	std::set<DynamicCodeLoader::Ptr>& dynamicCodeLoaders();
	
protected:
	Page(const std::string& name, const std::type_info& type);
		/// Creates a Page and assigns it the given name.
		
	Page(const std::type_info& type);
		/// Creates a Page.
		
	~Page();
		/// Destroys the Page.
		
private:
	std::string     _text;
	ResourceManager _rm;
	std::vector<std::string> _jsCode;
	std::set<DynamicCodeLoader::Ptr> _codeLoaders;
	std::string     _postRenderCode;
};


//
// Inlines
//

inline ResourceManager& Page::resourceManager()
{
	return _rm;
}

		
inline const ResourceManager& Page::resourceManager() const
{
	return _rm;
}


inline void Page::handleForm(const std::string& , const std::string& )
{
}


inline void Page::addDynamicFunction(const std::string& jsCode)
{
	_jsCode.push_back(jsCode);
}

		
inline const std::vector<std::string>& Page::dynamicFunctions() const
{
	return _jsCode;
}


inline void Page::setPostRenderCode(const std::string& js)
{
	_postRenderCode = js;
}

		
inline const std::string& Page::getPostRenderCode() const
{
	return _postRenderCode;
}


inline void Page::appendPostRenderCode(const std::string& js)
{
	_postRenderCode.append(js);
}


inline bool Page::serializeJSON(std::ostream&, const std::string&)
{
	return false;
}


inline void Page::addDynamicCodeLoader(DynamicCodeLoader::Ptr pLoader)
{
	if (pLoader) _codeLoaders.insert(pLoader);
}


inline const std::set<DynamicCodeLoader::Ptr>& Page::dynamicCodeLoaders() const
{
	return _codeLoaders;
}


inline std::set<DynamicCodeLoader::Ptr>& Page::dynamicCodeLoaders()
{
	return _codeLoaders;
}


} } // namespace Poco::WebWidgets


#endif // WebWidgets_Page_INCLUDED
