//
// ListBox.h
//
// $Id: //poco/Main/WebWidgets/include/Poco/WebWidgets/ListBox.h#2 $
//
// Library: WebWidgets
// Package: Controls
// Module:  ListBox
//
// Definition of the ListBox class.
//
// Copyright (c) 2007, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef WebWidgets_ListBox_INCLUDED
#define WebWidgets_ListBox_INCLUDED


#include "Poco/WebWidgets/Control.h"
#include "Poco/WebWidgets/ListBoxCell.h"
#include "Poco/WebWidgets/JavaScriptEvent.h"
#include "Poco/FIFOEvent.h"
#include <vector>
#include <set>


namespace Poco {
namespace WebWidgets {


class WebWidgets_API ListBox: public Control
	/// /// A ListBox allows multi selection from a given set of list entries
{
public:
	typedef Poco::AutoPtr<ListBox> Ptr;
	typedef ListBoxCell::Data Data; /// An element plus its selected flag as std::pair
	typedef ListBoxCell::RowValueChange RowValueChange; /// the row plus old and new value

	FIFOEvent<Poco::Net::HTTPServerResponse*> beforeLoad; /// thrown whenever a load is requested, internal event to which the ListBoxCellRenderer must register

	JavaScriptEvent<ListBox*> afterLoad; // thrown after data was loaded

	FIFOEvent<int> rowSelected; /// fires the row selected event

	FIFOEvent<int> rowDeselected; /// fires the row deselected event

	JavaScriptEvent<const std::set<int> > selectionChanged; /// fires the row selection change event

	ListBox(const std::string& name);
		/// Creates a ListBox with the given name.

	ListBox();
		/// Creates an anonymous TextField.

	Data::const_iterator begin() const;
		/// ConstIterator to all elements

	Data::iterator begin();
		/// Iterator to all elements

	Data::const_iterator end() const;
		/// ConstIterator to all elements

	Data::iterator end();
		/// Iterator to all elements
		
	void clearElements();
		/// Removes all elements

	void setElements(const Data& elems);
		/// Initializes the combo box with the provided elements
		
	template <typename T>
	void setElements(const std::vector<T>& elems)
		/// Initializes the combo box with the provided elements
		/// Elements are deselected by default
	{
		Data anyElems;
		typename std::vector<T>::const_iterator it = elems.begin();
		for (; it != elems.end(); ++it)
			anyElems.push_back(std::make_pair(Poco::Any(*it), false));
		setElements(anyElems);
	}		

	const Data& getElements() const;
		/// Returns all elements
		
	template <typename T>
	void getElements( std::vector<T>& data) const
		/// Returns the elements in param data
	{
		data.clear();
		const Data& elems = getElements();
		typename Data::const_iterator it = elems.begin();
		for (; it != elems.end(); ++it)
		{
			data.push_back(RefAnyCast<T>(it->first));
		}
	}


	void insert(const Any& elem, bool selected = false);
		/// Add a single element to the collection, not-selected per default

	void erase(const Any& elem);
		/// Removes a single element from the collection.

	void select(const Any& elem);
		/// Selects the element.

	void deselect(const Any& elem);
		/// Deselects the element.

	void selectByIndex(int idx, bool sel = true);
		/// Selects the element by Index. idx values that are out of range are ignored

	void deselectByIndex(int idx);
		/// Deselects the element. idx values that are out of range are ignored

	void selectAll(bool sel= true);
		/// Selects all elements

	void deselectAll();
		/// Deselects all elements

	bool hasSelected() const;
		/// Returns true if at least one selected element exists

	const Any& getSelected() const;
		/// Returns the first selected element, exception if none was selected
		
	void autoScroll(bool val = true);
		// sets autoscrolling
		
	bool autoScroll() const;
		/// gets autoscroll	

protected:
	ListBox(const std::string& name, const std::type_info& type);
		/// Creates a ListBox and assigns it the given name.

	ListBox(const std::type_info& type);
		/// Creates a ListBox.

	ListBox(const std::string& name, const std::type_info& type, ListBoxCell::Ptr ptrCell);
		/// Creates a ListBox and assigns it the given name.

	ListBox(const std::type_info& type, ListBoxCell::Ptr ptrCell);
		/// Creates a ListBox.

	~ListBox();
		/// Destroys the ListBox.

	void init();
		/// Common init code for all ctors.

	void init(Cell::Ptr pCell);
		/// Common init code for all ctors.

	void fireBeforeLoad(std::pair<ListBoxCell*, Poco::Net::HTTPServerResponse*>& data);

	void fireAfterLoad(void* pSender);

	void fireRowSelected(int& pos);

	void fireRowDeselected(int& pos);

	void fireSelectionChange(const std::set<int>& sel);

private:
	ListBoxCell* _pLBCell;
};


//
// inlines
//


inline bool ListBox::hasSelected() const
{
	return _pLBCell->hasSelected();
}


inline ListBox::Data::const_iterator ListBox::begin() const
{
	return _pLBCell->begin();
}


inline ListBox::Data::iterator ListBox::begin()
{
	return _pLBCell->begin();
}


inline ListBox::Data::const_iterator ListBox::end() const
{
	return _pLBCell->end();
}


inline ListBox::Data::iterator ListBox::end()
{
	return _pLBCell->end();
}


inline void ListBox::setElements(const ListBox::Data& elems)
{
	_pLBCell->setElements(elems);
}


inline const ListBox::Data& ListBox::getElements() const
{
	return _pLBCell->getElements();
}


inline void ListBox::insert(const Any& elem, bool selected)
{
	_pLBCell->insert(elem, selected);
}


inline void ListBox::erase(const Any& elem)
{
	_pLBCell->erase(elem);
}


inline void ListBox::select(const Any& elem)
{
	_pLBCell->select(elem);
}


inline void ListBox::deselect(const Any& elem)
{
	_pLBCell->deselect(elem);
}


inline const Any& ListBox::getSelected() const
{
	return _pLBCell->getSelected();
}


inline void ListBox::selectByIndex(int idx, bool sel)
{
	_pLBCell->selectByIndex(idx, sel);
}


inline void ListBox::deselectByIndex(int idx)
{
	_pLBCell->deselectByIndex(idx);
}


inline void ListBox::selectAll(bool sel)
{
	_pLBCell->selectAll(sel);
}


inline void ListBox::deselectAll()
{
	_pLBCell->deselectAll();
}


inline void ListBox::clearElements()
{
	_pLBCell->clearElements();
}


inline void ListBox::autoScroll(bool val)
{
	_pLBCell->autoScroll(val);
}

		
inline bool ListBox::autoScroll() const
{
	return _pLBCell->autoScroll();
}


} } // namespace Poco::WebWidgets


#endif // WebWidgets_ListBox_INCLUDED
