//
// JavaScriptEvent.h
//
// $Id: //poco/Main/WebWidgets/include/Poco/WebWidgets/JavaScriptEvent.h#7 $
//
// Library: WebWidgets
// Package: WebEvents
// Module:  JavaScriptEvent
//
// Definition of the JavaScriptEvent class.
//
// Copyright (c) 2007, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef WebWidgets_JavaScriptEvent_INCLUDED
#define WebWidgets_JavaScriptEvent_INCLUDED


#include "Poco/WebWidgets/WebWidgets.h"
#include "Poco/WebWidgets/JSDelegate.h"
#include "Poco/AbstractEvent.h"
#include "Poco/FIFOStrategy.h"
#include "Poco/AbstractDelegate.h"
#include "Poco/CompareFunctions.h"
#include <list>


namespace Poco {
namespace WebWidgets {


enum ServerCallback
{
	SC_NO = 0,
	SC_YES,
	SC_ONLOCALHANDLERS
};

	
template <class TArgs> 
class JavaScriptEvent: public Poco::AbstractEvent < 
	TArgs, Poco::FIFOStrategy<TArgs, Poco::AbstractDelegate<TArgs>, Poco::p_less<Poco::AbstractDelegate<TArgs> > >,
	Poco::AbstractDelegate<TArgs> 
>
	/// Event class used to handle JavaScriptEvents. Allows to register two different types
	/// of delegates. The standard delegates, as known from Poco::FIFOEvent and JSDelegates
	/// which will be embedded into the WebPage when the Parser generates the site.
	/// Per default a server callback happens only when local listeners are registered.
{
public:
	typedef typename std::list<JSDelegate> JSDelegates;
	
	JavaScriptEvent():
		_jsHandlers(),
		_serverCallback(SC_ONLOCALHANDLERS),
		_callbackPos(0),
		_onSuccess(),
		_onFailure()
		/// Creates the JavaScriptEvent.
	{
	}

	~JavaScriptEvent()
		/// Destroys the JavaScriptEvent.
	{
	}

	bool hasLocalHandlers () const
	{
		FastMutex::ScopedLock lock(this->_mutex);
		return !this->_strategy.empty();
	}

	bool hasJavaScriptHandlers() const
	{
		FastMutex::ScopedLock lock(this->_mutex);
		return !_jsHandlers.empty();
	}

	void add (const JSDelegate& aDelegate)
		/// Adds a javascript delegate to the event.
	{
		FastMutex::ScopedLock lock(this->_mutex);
		_jsHandlers.push_back(aDelegate);
	}
	
	void remove (const JSDelegate& aDelegate)
		/// Removes a javascript delegate from the event. 
		/// If the observer is not found, the unregister will be ignored
	{
		FastMutex::ScopedLock lock(this->_mutex);
		_jsHandlers.remove(aDelegate);
	}

	const JSDelegates& jsDelegates() const
		/// Returns all delegates currently registered
	{
		return _jsHandlers;
	}
	
	void setJSDelegates(const JSDelegates& all)
		/// Overwrites all JSDelegates
	{
		_jsHandlers = all;
	}
	
	void setServerCallback(ServerCallback sc, const std::string& onSuccess = "", const std::string& onFailure = "")
		/// Sets the server callback. onSuccess and onFailure must either be a JavaScript function pointer or a complete
		/// anonymous function. The server callback position will be set after the currently defined jsHandlers.
		/// If you want to set the server callback as the very first jsHandler, call setServerCallback before
		/// any event->add.
		/// Note that with most renderers you should not worry about the position of the server callback due to its 
		/// asynchronous nature! If you require to execute JS code after the server callback, you must use onSuccess or
		/// onFailure!
	{
		FastMutex::ScopedLock lock(this->_mutex);
		_serverCallback = sc;
		_callbackPos = _jsHandlers.size();
		_onSuccess = onSuccess;
		_onFailure = onFailure;
	}
	
	bool willDoServerCallback() const
		/// Checks if servercallback is set to true or to DEPENDS and if localHandlers exist
	{
		return (_serverCallback == SC_YES || (_serverCallback == SC_ONLOCALHANDLERS && hasLocalHandlers()));
	}

	ServerCallback getServerCallback() const
		/// Returns the server callback value
	{
			return _serverCallback;
	}
	
	std::size_t getServerCallbackPos() const
		/// Returns how many jsHandlers are executed before the serverCallback
	{
		return _callbackPos;
	}
	
	void setServerCallbackPos(std::size_t pos) const
		/// Sets how many jsHandlers are executed before the serverCallback. Set to 0 to guarantee that the server callback is the first
	{
		_callbackPos = pos;
	}
	
	const std::string& getOnSuccess() const
		/// Returns the onSuccess value
		
	{
		return _onSuccess;
	}
	
	const std::string& getOnFailure() const
		/// Returns the onFailure value
	{
		return _onFailure;
	}
	
	bool hasJavaScriptCode() const
		/// Returns true if any JSCode (incl, the server callback) is attached to this event
	{
		return (willDoServerCallback() || !_jsHandlers.empty());
	}
	
	void setDelayTime(int millisecs)
		/// Sets the delay time for the javascript callback
	{
		_delayTime = millisecs;
	}
	
	int getDelayTime() const
		/// Returns the delay time in milliseconds for the javascript callback
	{
		return _delayTime;
	}
	
	bool getGroupEvents() const
		/// Returns if events should be grouped together within the given _delayTime.
		/// Every event that occurs within the tiem frame replaces the previous scheduled event
		/// and restart the _delayTime wait.
	{
		return _group;
	}
	
	void setGroupEvents(bool group)
		/// Sets if events should be grouped together within the given _delayTime.
		/// Every event that occurs within the tiem frame replaces the previous scheduled event
		/// and restart the _delayTime wait.
	{
		_group = group;
	}
		
private:
	JSDelegates    _jsHandlers;
	ServerCallback _serverCallback; /// Set to SC_YES if a server callback should be done always
	std::size_t    _callbackPos;    /// Sets when the server callback should happen (the number defines how many jsHandlers are executed before the server callback)
	std::string    _onSuccess;      /// code to execute when the server callback succeeds
	std::string    _onFailure;      /// code to execute when the server callback fails
	int            _delayTime;		/// delay Time in millisecs
	bool           _group;			
		/// defines if events should be grouped together. 
		/// Requires a _delayTime greater zero, replaces an event within the delaytime with its newer version
		/// Only effective for Javascript side of the event!
};


} } // namespace Poco::WebWidgets


#endif // WebWidgets_JavaScriptEvent_INCLUDED
