//
// Cell.h
//
// $Id: //poco/Main/WebWidgets/include/Poco/WebWidgets/Cell.h#3 $
//
// Library: WebWidgets
// Package: Core
// Module:  Cell
//
// Definition of the Cell class.
//
// Copyright (c) 2008, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef WebWidgets_Cell_INCLUDED
#define WebWidgets_Cell_INCLUDED


#include "Poco/WebWidgets/Renderable.h"
#include "Poco/WebWidgets/Formatter.h"
#include "Poco/WebWidgets/View.h"
#include "Poco/WebWidgets/RequestProcessor.h"
#include "Poco/Any.h"


namespace Poco {
namespace WebWidgets {


class WebWidgets_API Cell: public Renderable, public RequestProcessor
	/// Cell is a lightweight class that works in conjunction with a Control or 
	/// Matrix. A Cell does not provide events (these are provided by the 
	/// Controller owning the Cell). However, a Cell processes requests from 
	/// the Web browser. Thus, cells register themselves for HTTP requests with 
	/// the WidgetRequestHandler.
	///
	/// Cells can react to HTTP requests from the browser in two ways - by 
	/// handling a HTML Form item, or by handling a complete request (Ajax-style).
	/// To handle HTTP requests, a Cell registers itself (or is registered by
	/// its Renderer) with the HTTP server during the rendering process.
{
public:
	typedef Poco::AutoPtr<Cell> Ptr;
	
	enum EditMode
		/// Confgures edit mode for cells
	{
		EM_SELECTCONTENT = 0,
		EM_CURSORATENDOFVALUE
	};

	void enable(bool enabled = true);
		/// Enables (if enabled == true) or disables the Cell.
		
	void disable();
		/// Disables the Cell.
		
	bool isEnabled() const;
		/// Returns true iff the Cell is enabled.

	void setToolTip(const std::string& text);
		/// Sets the tooltip text for this Cell.
		
	const std::string& getToolTip() const;
		/// Returns the tooltip text for this Cell.
		
	void setValue(const Poco::Any& value);
		/// Sets the value for this Cell.
		///
		/// The actual type of the value must match the type
		/// expected by the Cell or its Formatter.
		
	const Poco::Any& getValue() const;
		/// Returns the value for this Cell.

	void setValue(const std::string& value);
		/// Sets the value by converting the given string using
		/// the Formatter. If no Formatter has been set, sets the value
		/// to the given string.
		
	bool hasValue() const;
		
	void setString(const std::string& value);	
		/// Sets the value for this Cell to the given string.
		
	std::string getString() const;
		/// Returns the value of this Cell formatted as a string.
		///
		/// Throws a BadCastException if the Cell's value is not a string and
		/// no formatter has been set.
		
	void setInt(int value);
		/// Sets the value of this Cell to the given int value.
		
	int getInt() const;
		/// Returns the value of this Cell as an int value.
		///
		/// Throws a BadCastException if the Cell's value is not an int.

	void setDouble(double value);
		/// Sets the value of this Cell to the given double value.
		
	double getDouble() const;
		/// Returns the value of this Cell as a double value.
		///
		/// Throws a BadCastException if the Cell's value is not a double.

	void setBool(bool value);
		/// Sets the value of this Cell to the given bool value.
		
	bool getBool() const;
		/// Returns the value of this Cell as a bool value.
		///
		/// Throws a BadCastException if the Cell's value is not a bool.

	void setFormatter(Formatter::Ptr pFormatter);
		/// Sets the Formatter for this Cell.
		
	Formatter::Ptr getFormatter() const;
		/// Returns the Formatter for this Cell.
	
	void setOwner(View* pOwner);
		/// Sets the owner of this Cell.
	
	View* getOwner() const;
		/// Returns the owner of this Cell.
		
	void setRowIndex(int rowIndex);
		/// Sets the Cell's row index (for Cells used in a TableColumn).
		
	int getRowIndex() const;
		/// Returns the Cell's row index (for Cells used in a TableColumn).
		///
		/// For Cells that are not part of a TableColumn, the row index is
		/// set to -1.

	void setEditable(bool editable = true);
		/// Sets whether the Cell's contents can be edited.

	bool isEditable() const;
		/// Returns true iff the Cell is editable.
		
	void setEditMode(Cell::EditMode em);
		/// Sets the edit mode
		
	Cell::EditMode getEditMode() const;	
		/// Returns the edit mode	
	
	void setClass(const std::string cls);
		/// Sets css class
		
	const std::string& getClass() const;
		/// Returns css class
		
	bool hasClass() const;
		/// Checks if a css value is set	
		
	// RequestProcessor
	virtual void handleForm(const std::string& field, const std::string& value);
		/// Handles a form field submitted by the client.
		///
		/// The default implementation does nothing.
	
	virtual void handleAjaxRequest(const Poco::Net::NameValueCollection& args, Poco::Net::HTTPServerResponse& response);
		/// Handles a complete AJAX request submitted by the client.
		///
		/// The default implementation does nothing and sends an HTTP Status 200 OK

protected:
	Cell(View* pOwner, const std::type_info& type);
		/// Creates a Cell.
		
	Cell(const std::type_info& type);
		/// Creates a Cell.
		
	~Cell();
		/// Destroys the Cell.
		
	View& owner() const;
		/// Returns the owner of this Cell.
		
private:
	EditMode       _em;
	View*          _pOwner;
	bool           _enabled;
	bool           _editable;
	std::string    _toolTip;
	Poco::Any      _value;
	Formatter::Ptr _pFormatter;
	int            _rowIndex;
	std::string    _class;
};


//
// inlines
//
inline bool Cell::isEnabled() const
{
	return _enabled;
}


inline void Cell::disable()
{
	enable(false);
}


inline bool Cell::isEditable() const
{
	return _editable;
}


inline void Cell::setEditable(bool editable)
{
	_editable = editable;
}


inline const std::string& Cell::getToolTip() const
{
	return _toolTip;
}


inline const Poco::Any& Cell::getValue() const
{
	return _value;
}


inline Formatter::Ptr Cell::getFormatter() const
{
	return _pFormatter;
}


inline View* Cell::getOwner() const
{
	return _pOwner;
}


inline View& Cell::owner() const
{
	poco_check_ptr (_pOwner);
	
	return *_pOwner;
}


inline int Cell::getRowIndex() const
{
	return _rowIndex;
}


inline void Cell::setEditMode(Cell::EditMode em)
{
	_em = em;
}

		
inline Cell::EditMode Cell::getEditMode() const
{
	return _em;
}


inline bool Cell::hasValue() const
{
	return !getValue().empty();
}


inline void Cell::setClass(const std::string cls)
{
	_class = cls;
}

		
inline const std::string& Cell::getClass() const
{
	return _class;
}

		
inline bool Cell::hasClass() const
{
	return !getClass().empty();
}


} } // namespace Poco::WebWidgets


#endif // WebWidgets_Cell_INCLUDED
