//
// SecureSocketImpl.h
//
// $Id: //poco/Main/NetSSL_OpenSSL/include/Poco/Net/SecureSocketImpl.h#8 $
//
// Library: NetSSL_OpenSSL
// Package: SSLSockets
// Module:  SecureSocketImpl
//
// Definition of the SecureSocketImpl class.
//
// Copyright (c) 2006-2009, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef NetSSL_SecureSocketImpl_INCLUDED
#define NetSSL_SecureSocketImpl_INCLUDED


#include "Poco/Net/NetSSL.h"
#include "Poco/Net/SocketImpl.h"
#include "Poco/Net/Context.h"
#include "Poco/Net/X509Certificate.h"
#include <openssl/bio.h>
#include <openssl/ssl.h>


namespace Poco {
namespace Net {


class HostEntry;


class NetSSL_API SecureSocketImpl
	/// The SocketImpl for SecureStreamSocket.
{
public:
	SecureSocketImpl(Poco::AutoPtr<SocketImpl> pSocketImpl, Context::Ptr pContext);
		/// Creates the SecureSocketImpl using an already
		/// connected stream socket.

	virtual ~SecureSocketImpl();
		/// Destroys the SecureSocketImpl.

	SocketImpl* acceptConnection(SocketAddress& clientAddr);
		/// Get the next completed connection from the
		/// socket's completed connection queue.
		///
		/// If the queue is empty, waits until a connection
		/// request completes.
		///
		/// Returns a new SSL socket for the connection
		/// with the client.
		///
		/// The client socket's address is returned in clientAddr.
	
	void acceptSSL();
		/// Performs a server-side SSL handshake and certificate verification.
	
	void connect(const SocketAddress& address, const std::string& hostName);
		/// Initializes the socket and establishes a secure connection to 
		/// the TCP server at the given address.

	void connect(const SocketAddress& address, const std::string& hostName, const Poco::Timespan& timeout);
		/// Initializes the socket, sets the socket timeout and 
		/// establishes a secure connection to the TCP server at the given address.

	void connectNB(const SocketAddress& address, const std::string& hostName);
		/// Initializes the socket and establishes a secure connection to 
		/// the TCP server at the given address. Prior to opening the
		/// connection the socket is set to nonblocking mode.

	void connectSSL(const std::string& hostName);
		/// Performs a client-side SSL handshake and establishes a secure 
		/// connection over an already existing TCP connection.
	
	void bind(const SocketAddress& address, bool reuseAddress = false);
		/// Bind a local address to the socket.
		///
		/// This is usually only done when establishing a server
		/// socket. SSL clients should not bind a socket to a
		/// specific address.
		///
		/// If reuseAddress is true, sets the SO_REUSEADDR
		/// socket option.
		
	void listen(int backlog = 64);
		/// Puts the socket into listening state.
		///
		/// The socket becomes a passive socket that
		/// can accept incoming connection requests.
		///
		/// The backlog argument specifies the maximum
		/// number of connections that can be queued
		/// for this socket.

	void shutdown();
		/// Shuts down the connection by attempting
		/// an orderly SSL shutdown, then actually
		/// shutting down the TCP connection.

	void close();
		/// Close the socket.
	
	int sendBytes(const void* buffer, int length, int flags = 0);
		/// Sends the contents of the given buffer through
		/// the socket. Any specified flags are ignored.
		///
		/// Returns the number of bytes sent, which may be
		/// less than the number of bytes specified.
	
	int receiveBytes(void* buffer, int length, int flags = 0);
		/// Receives data from the socket and stores it
		/// in buffer. Up to length bytes are received.
		///
		/// Returns the number of bytes received.
	
	poco_socket_t sockfd();
		/// Returns the underlying socket descriptor.

	X509* peerCertificate() const;
		/// Returns the peer's certificate.
		
	Context::Ptr context() const;
		/// Returns the SSL context used for this socket.

protected:
	long verifyCertificate(const std::string& hostName);
		/// PostConnectionCheck to verify that a peer really presented a valid certificate.
		
	static bool isLocalHost(const std::string& hostName);
		/// Returns true iff the given host name is the local host 
		/// (either "localhost" or "127.0.0.1").
		
	int handleError(int rc);
		/// Handles an SSL error by throwing an appropriate exception.

private:	
	SecureSocketImpl(const SecureSocketImpl&);
	SecureSocketImpl& operator = (const SecureSocketImpl&);

	SSL* _pSSL;
	Poco::AutoPtr<SocketImpl> _pSocket;
	Context::Ptr _pContext;
};


//
// inlines
//
inline poco_socket_t SecureSocketImpl::sockfd()
{
	return _pSocket->sockfd();
}


inline Context::Ptr SecureSocketImpl::context() const
{
	return _pContext;
}


} } // namespace Poco::Net


#endif // NetSSL_SecureSocketImpl_INCLUDED
