//
// SMTPChannel.h
//
// $Id: //poco/svn/Net/include/Poco/Net/SMTPChannel.h#1 $
//
// Library: Net
// Package: Logging
// Module:  SMTPChannel
//
// Definition of the SMTPChannel class.
//
// Copyright (c) 2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Net_SMTPChannel_INCLUDED
#define Net_SMTPChannel_INCLUDED


#include "Poco/Net/Net.h"
#include "Poco/Channel.h"
#include "Poco/String.h"


namespace Poco {
namespace Net {


class Net_API SMTPChannel: public Poco::Channel
	/// This Channel implements SMTP (email) logging.
{
public:
	SMTPChannel();
		/// Creates a SMTPChannel.
		
	SMTPChannel(const std::string& mailhost, const std::string& sender, const std::string& recipient);
		/// Creates a SMTPChannel with the given target mailhost, sender, and recipient.
	
	void open();
		/// Opens the SMTPChannel.
		
	void close();
		/// Closes the SMTPChannel.
		
	void log(const Message& msg);
		/// Sends the message's text to the recipient.
		
	void setProperty(const std::string& name, const std::string& value);
		/// Sets the property with the given value.
		///
		/// The following properties are supported:
		///     * mailhost:   The SMTP server. Default is "localhost".
		///     * sender:     The sender address.
		///     * recipient:  The recipient address.
		///     * local:      If true, local time is used. Default is true.
		///     * attachment: Filename of the file to attach.
		///     * type:       Content type of the file to attach.
		///     * delete:     Boolean value indicating whether to delete 
		///                   the attachment file after sending.
		///     * throw:      Boolean value indicating whether to throw 
		///                   exception upon failure.
		
	std::string getProperty(const std::string& name) const;
		/// Returns the value of the property with the given name.

	static void registerChannel();
		/// Registers the channel with the global LoggingFactory.

	static const std::string PROP_MAILHOST;
	static const std::string PROP_SENDER;
	static const std::string PROP_RECIPIENT;
	static const std::string PROP_LOCAL;
	static const std::string PROP_ATTACHMENT;
	static const std::string PROP_TYPE;
	static const std::string PROP_DELETE;
	static const std::string PROP_THROW;

protected:
	~SMTPChannel();

private:
	bool isTrue(const std::string& value) const;

	std::string _mailHost;
	std::string _sender;
	std::string _recipient;
	bool        _local;
	std::string _attachment;
	std::string _type;
	bool        _delete;
	bool        _throw;
};


inline bool SMTPChannel::isTrue(const std::string& value) const
{
	return ((0 == icompare(value, "true")) ||
			(0 == icompare(value, "t")) ||
			(0 == icompare(value, "yes")) ||
			(0 == icompare(value, "y")));
}


} } // namespace Poco::Net


#endif // Net_SMTPChannel_INCLUDED
