//
// HashTableTest.cpp
//
// $Id: //poco/svn/Foundation/testsuite/src/HashTableTest.cpp#2 $
//
// Copyright (c) 2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#include "HashTableTest.h"
#include "CppUnit/TestCaller.h"
#include "CppUnit/TestSuite.h"
#include "Poco/HashTable.h"
#include "Poco/NumberFormatter.h"


using namespace Poco;


HashTableTest::HashTableTest(const std::string& name): CppUnit::TestCase(name)
{
}


HashTableTest::~HashTableTest()
{
}


void HashTableTest::testInsert()
{
	std::string s1("str1");
	std::string s2("str2");
	HashTable<std::string, int> hashTable;
	assert (!hashTable.exists(s1));
	hashTable.insert(s1, 13);
	assert (hashTable.exists(s1));
	assert (hashTable.get(s1) == 13);
	int retVal = 0;

	assert (hashTable.get(s1, retVal));
	assert (retVal == 13);
	try
	{
		hashTable.insert(s1, 22);
		failmsg ("duplicate insert must fail");
	}
	catch (Exception&){}
	try
	{
		hashTable.get(s2);
		failmsg ("getting a non inserted item must fail");
	}
	catch (Exception&){}

	assert (!hashTable.exists(s2));
	hashTable.insert(s2, 13);
	assert (hashTable.exists(s2));
}


void HashTableTest::testUpdate()
{
	// add code for second test here
	std::string s1("str1");
	std::string s2("str2");
	HashTable<std::string, int> hashTable;
	hashTable.insert(s1, 13);
	hashTable.update(s1, 14);
	assert (hashTable.exists(s1));
	assert (hashTable.get(s1) == 14);
	int retVal = 0;

	assert (hashTable.get(s1, retVal));
	assert (retVal == 14);

	// updating a non existing item must work too
	hashTable.update(s2, 15);
	assert (hashTable.get(s2) == 15);
}


void HashTableTest::testOverflow()
{
	HashTable<std::string, int> hashTable(13);
	for (int i = 0; i < 1024; ++i)
	{
		hashTable.insert(Poco::NumberFormatter::format(i), i*i);
	}

	for (int i = 0; i < 1024; ++i)
	{
		std::string tmp = Poco::NumberFormatter::format(i);
		assert (hashTable.exists(tmp));
		assert (hashTable.get(tmp) == i*i);
	}
}


void HashTableTest::testSize()
{
	HashTable<std::string, int> hashTable(13);
	assert (hashTable.size() == 0);
	Poco::UInt32 h1 = hashTable.insert("1", 1);
	assert (hashTable.size() == 1);
	Poco::UInt32 h2 = hashTable.update("2", 2);
	assert (hashTable.size() == 2);
	hashTable.remove("1");
	assert (hashTable.size() == 1);
	hashTable.remove("3");
	assert (hashTable.size() == 1);
	hashTable.removeRaw("2", h2);
	assert (hashTable.size() == 0);
	hashTable.insert("1", 1);
	hashTable.insert("2", 2);
	assert (hashTable.size() == 2);
	hashTable.clear();
	assert (hashTable.size() == 0);
}


void HashTableTest::testResize()
{
	HashTable<std::string, int> hashTable(13);
	assert (hashTable.size() == 0);
	hashTable.resize(19);
	for (int i = 0; i < 1024; ++i)
	{
		hashTable.insert(Poco::NumberFormatter::format(i), i*i);
	}
	hashTable.resize(1009);

	for (int i = 0; i < 1024; ++i)
	{
		std::string tmp = Poco::NumberFormatter::format(i);
		assert (hashTable.exists(tmp));
		assert (hashTable.get(tmp) == i*i);
	}
}


void HashTableTest::testStatistic()
{
	double relax = 0.001;
	HashTable<std::string, int> hashTable(13);
	assert (hashTable.size() == 0);
	HashStatistic stat1(hashTable.currentState());
	assert (stat1.avgEntriesPerHash() < relax && stat1.avgEntriesPerHash() > -relax);
	assert (stat1.maxPositionsOfTable() == 13);
	assert (stat1.maxEntriesPerHash() == 0);

	hashTable.resize(19);
	stat1 = hashTable.currentState(true);
	assert (stat1.avgEntriesPerHash() < relax && stat1.avgEntriesPerHash() > -relax);
	assert (stat1.maxPositionsOfTable() == 19);
	assert (stat1.maxEntriesPerHash() == 0);
	assert (stat1.detailedEntriesPerHash().size() == 19);

	for (int i = 0; i < 1024; ++i)
	{
		hashTable.insert(Poco::NumberFormatter::format(i), i*i);
	}
	stat1 = hashTable.currentState(true);
	double expAvg = 1024.0/ 19;
	assert (stat1.avgEntriesPerHash() < (expAvg + relax) && stat1.avgEntriesPerHash() > (expAvg - relax));
	assert (stat1.maxPositionsOfTable() == 19);
	assert (stat1.maxEntriesPerHash() > expAvg);
	hashTable.resize(1009);
	stat1 = hashTable.currentState(true);

	expAvg = 1024.0/ 1009;

	assert (stat1.avgEntriesPerHash() < (expAvg + relax) && stat1.avgEntriesPerHash() > (expAvg - relax));
	assert (stat1.maxPositionsOfTable() == 1009);
	assert (stat1.maxEntriesPerHash() > expAvg);
}


void HashTableTest::setUp()
{
}


void HashTableTest::tearDown()
{
}


CppUnit::Test* HashTableTest::suite()
{
	CppUnit::TestSuite* pSuite = new CppUnit::TestSuite("HashTableTest");

	CppUnit_addTest(pSuite, HashTableTest, testInsert);
	CppUnit_addTest(pSuite, HashTableTest, testUpdate);
	CppUnit_addTest(pSuite, HashTableTest, testOverflow);
	CppUnit_addTest(pSuite, HashTableTest, testSize);
	CppUnit_addTest(pSuite, HashTableTest, testResize);
	CppUnit_addTest(pSuite, HashTableTest, testStatistic);

	return pSuite;
}
