//
// SessionPoolContainer.cpp
//
// $Id: //poco/Main/Data/src/SessionPoolContainer.cpp#3 $
//
// Library: Data
// Package: SessionPoolContainering
// Module:  SessionPoolContainer
//
// Copyright (c) 2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#include "Poco/Data/SessionPoolContainer.h"
#include "Poco/Data/SessionFactory.h"
#include "Poco/Data/DataException.h"
#include "Poco/URI.h"
#include "Poco/String.h"
#include "Poco/Exception.h"
#include <algorithm>


namespace Poco {
namespace Data {


SessionPoolContainer::SessionPoolContainer()
{
}


SessionPoolContainer::~SessionPoolContainer()
{
}


void SessionPoolContainer::add(SessionPool* pPool)
{
	poco_check_ptr (pPool);

	if (_sessionPools.find(pPool->name()) != _sessionPools.end())
		throw SessionPoolExistsException("Session pool already exists: " + pPool->name());

	pPool->duplicate();
	_sessionPools.insert(SessionPoolMap::ValueType(pPool->name(), pPool));
}


Session SessionPoolContainer::add(const std::string& sessionKey, 
	const std::string& connectionString,
	int minSessions, 
	int maxSessions, 
	int idleTime)
{
	std::string name = SessionPool::name(sessionKey, connectionString);
	SessionPoolMap::Iterator it = _sessionPools.find(name);

	// pool already exists, silently return a session from it
	if (it != _sessionPools.end()) return it->second->get();

	SessionPool* pSP = 
		new SessionPool(sessionKey, connectionString, minSessions, maxSessions, idleTime);

	std::pair<SessionPoolMap::Iterator, bool> ins = 
		_sessionPools.insert(SessionPoolMap::ValueType(name, pSP));

	return ins.first->second->get();
}


Session SessionPoolContainer::get(const std::string& name)
{
	URI uri(name);
	std::string path = uri.getPath();
	poco_assert (!path.empty());
	std::string n = Session::uri(uri.getScheme(), path.substr(1));
	SessionPoolMap::Iterator it = _sessionPools.find(n);
	if (_sessionPools.end() == it) throw NotFoundException(n);
	return it->second->get();
}


void SessionPoolContainer::shutdown()
{
	SessionPoolMap::Iterator it = _sessionPools.begin();
	SessionPoolMap::Iterator end = _sessionPools.end();
	for (; it != end; ++it) it->second->shutdown();
}


} } // namespace Poco::Data
