/**
 * @file TestSched.cpp
 * @brief Tests for ODEMx class Sched
 */

#include "TestBase.h"

/// @cond DOXYGEN_SKIP
SUITE( Base )
{
/// @endcond

	/**
	 * @struct SchedFixture
	 * @brief Helper struct providing set-up/tear-down of Sched tests
	 *
	 * @copydetails EventFixture
	 */
	struct SchedFixture
	{
		SchedFixture():
			sched1( getDefaultSimulation(), "SchedTest1", Sched::PROCESS, 1, 1 ),
			sched2( getDefaultSimulation(), "SchedTest2", Sched::EVENT, 2, 2 ),
			sched3( getDefaultSimulation(), "SchedTest3", Sched::EVENT, 2, 1 )
			{}

		SchedTest sched1; ///< a Sched test object
		SchedTest sched2; ///< a Sched test object
		SchedTest sched3; ///< a Sched test object
	};

	/**
	 * @test odemx::Sched construction and destruction
	 *
	 * Expected observations:
	 * @li SchedType is either Sched::PROCESS or Sched::EVENT
	 * @li Sched::isScheduled returns \c false
	 */
	TEST_FIXTURE( SchedFixture, ConstructionDestruction )
	{
		CHECK( sched1.getSchedType() == Sched::PROCESS );
		CHECK( sched2.getSchedType() == Sched::EVENT );
		CHECK( sched1.isScheduled() == false );
		CHECK( sched2.isScheduled() == false );
	}

	/**
	 * @test odemx::operator<(Sched*,Sched*)
	 *
	 * The three Sched objects are compared by their execution times and
	 * priority.
	 *
	 * Expected observations:
	 * @li \c sched1 comes before \c sched2 due to earlier execution time
	 * @li \c sched2 comes before \c sched3 despite the same execution time
	 * due to its higher priority
	 */
	TEST_FIXTURE( SchedFixture, OperatorLess )
	{
		CHECK( sched1.getExecutionTime() < sched2.getExecutionTime() );
		CHECK( sched1 < sched2 );
		CHECK( sched2.getPriority() > sched3.getPriority() );
		CHECK( sched2 < sched3 );
	}

	/**
	 * @test odemx::Sched::getTime
	 * A Sched object must return the current time when calling Sched::getTime().
	 *
	 * Expected observations:
	 * @li \c sched1 returns the result of \c Simulation::getTime()
	 * @li \c sched2 returns the result of \c Simulation::getTime()
	 * */
	TEST_FIXTURE( SchedFixture, GetTime )
	{
		CHECK( sched1.getTime() == sched1.getSimulation().getTime() );
		CHECK( sched2.getTime() == sched1.getSimulation().getTime() );
	}

/// @cond DOXYGEN_SKIP
}
/// @endcond
