//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file MonitorTest.cpp

 \author Michael Fiedler
 \author Sascha Qualitz
 <!-- [\author <author>]* -->

 \date created at 2008/11/21

 <!-- [detailed description] -->

 <!-- [\todo {todos for this file}]* -->

 \since 2.2
 */
/** \example Example_Monitor.cpp

	The furnace/ingot-example is to show continuous simulation techniques of ODEMx.
	This example describes a simulation of a furnace which heats ingots to the temperature of
	380 C. After reaching this temperature they leave the furnace. The temperature outside the furnace is 25 C
	and so the furnace have to heat up an ingot from 25 C to 380 C.
	After leaving the furnace the ingot cools down again to 25 C.

	Apart from discrete processes a simulation can also contain so called
	continuous processes. A discrete process changes the system state
	only at discrete moments in simulation time. It uses the
	actions defined in its 'main()' function. The progress of time for a
	process is realized with functions like 'holdFor', 'activateAt' and so on.
	See the basic simulation example for details.

	A continuous process instead changes the state of a system continuously.
	Such a process could be for example the melting of a block of metal
	inside an oven.
	At every time, during the melting process, the temperature of the
	metal changes. That means every time one measures the temperature one gets a new value.
	Of course real continuity is quite impossible in the discrete world of
	our computers. So ODEMx has to approximate continuous state changes with
	a step by step computation. This is realized by a monitor concept. The monitor controls all
	relevant objects during the integration. The relevant objects are Continuous, ODEObject,
	StateEvent and Process.
*/

#include <odemx/odemx.h>

using namespace odemx::base;
using namespace odemx::base::continuous;
using namespace std;

#ifdef ODEMX_USE_CONTINUOUS

// shows the number of ingots in the furnace
int inFurnace = 0;

/** equation for heating and cooling an ingot
 *  To define how the state changes during the time, every continuous object needs a set of equations.
 *  One has to rewrite the derivates(SimTime time)-function to define an equation.
 *  This is done by writting in an ordinary differential equation which describes the state change during the simulation.
 *  To do this one has to define the rate variables of type Rate and the state variables of type State.
 *  Note: Rate and State are C++-classes, so they have to be initialized with an existing continuous object.
 *  There can be more than one state variable, but every state/rate variable has a dimension, which is implicit set by the corresponding continuous object.
 *  They are accessed by an index, which lies between 0 and dimension - 1.
 *
 *  Note: The Rate/State objects only encapsulate the access to the values of one continuous object.
 *
 *  Also one needs to rewrite the jacobi(SimTime time)-function to define the jacobimatrix and for the value for differentiating
 *  the i-th component of f by time (if it is an autonomic system of ordinary differential equations it can be zero). Here one has to define jacobimatrix objects
 *  of type JacobiMatrix and dfdt objects of type DfDt.
 *  Note: JacobiMatrix and DfDt are C++-classes too. They have an index and one has to initialize the classes with the corresponding continuous objects.
 *
 *  How this is done is shown by the below following example of the class IngotEqn.
 */
class IngotEqn : public ODEObject {
public:
	//constructor
	IngotEqn (continuous::Continuous* continuous, continuous::Continuous* furnace)
	:	furnace(furnace)
	,	state(continuous)
	,	rate(continuous)
	,	stateFurnace(furnace)
	,	jacobiMatrix(continuous)
	,	furnaceJacobiMatrix(continuous, furnace)
	,	dfdt(continuous)
	,	ingotInFurnace(false)
	{
		this->continuous = continuous;
	}

	void derivates(SimTime time) {
		if (continuous == 0) {
			throw new NotAssignedException("Continous", "IngotEqn");
		}
		//if the ingot is inside the furnace, the boolean variable 'ingotInFurnace' is true.
		if (this->ingotInFurnace) {//in furnace
			if (furnace == 0) {
				throw new NotAssignedException("Continous", "IngotEqn");
			}

			// x'(t) = (u(t) - x(t)) / 7
			rate[0] = (stateFurnace[0] - state[0]) / 7;

		} else {//outside the furnace

			// x'(t) = (25.0 - x(t)) / 7
			rate[0] = (25.0 - state[0]) / 7;
		}
	}

	void jacobi(SimTime time) {
		if (continuous == 0) {
			throw new NotAssignedException("Continous", "IngotEqn");
		}
		if (this->ingotInFurnace) {
			if (furnace == 0) {
				throw new NotAssignedException("Continous", "IngotEqn");
			}
			// Df/Du = 1/7
			// Df/Dx = -1/7
			furnaceJacobiMatrix(0,0) = 1.0/7.0;
			jacobiMatrix(0,0) = -1.0 / 7.0;
		} else {
			// Df/Dx = -1/7
			jacobiMatrix(0,0) = -1.0 / 7.0;
		}

		//DfDt is zero as system is autonomic
		dfdt[0] = 0;
	}

	// if the ingot is inside the furnace 'ingotInFurnace' is set to true.
	void inFurnace(bool inFurnace) {
		this->ingotInFurnace = inFurnace;
	}

private:
	continuous::Continuous* furnace;
	State state;
	Rate rate;
	State stateFurnace;
	JacobiMatrix jacobiMatrix, furnaceJacobiMatrix;
	DfDt dfdt;
	bool ingotInFurnace;
};

/** equation for heating a furnace without ingot
 * 	While the furnace is empty, this equation describes the temperature of the furnace.
 *
 *  Like above one has to rewrite the derivates/jacobi-functions.
 */
class FurnaceEqn : public ODEObject {
public:
	//constructor
	FurnaceEqn(continuous::Continuous* continuous)
	:	rate(continuous)
	,	state(continuous)
	,	jacobiMatrix(continuous)
	,	dfdt(continuous)
	{

	}

	void derivates(SimTime time) {
		if (continuous == 0) {
			throw new NotAssignedException("Continous", "FurnaceEqn");
		}
		// u'(t) = 500 - u(t)
		rate[0] = 500 - state[0];
	}

	void jacobi(SimTime time) {
		if (continuous == 0) {
			throw new NotAssignedException("Continous", "FurnaceEqn");
		}
		// Df/Du = -1
		jacobiMatrix(0,0) = -1.0;

		//DfDt is zero as system is autonomic
		dfdt[0] = 0;
	}

private:
	Rate rate;
	State state;
	JacobiMatrix jacobiMatrix;
	DfDt dfdt;
};

/** equation for temperature change of the furnace affected by an ingot
 *
 * 	Like above one has to rewrite the derivates/jacobi-functions.
 *
 * 	As 'FurnaceIngotEqn'-object has no 'furnace'-object while it is initialized, the 'rate'-, 'state'- and 'ingotJacobiMatrix'-objects
 * 	also have no 'furnace'-object while initialized.
 *
 *	So the 'furnace'-object has to be set by the 'setContinouous(continuous)'-function of the corresponding 'Rate'-, 'State'- or 'JacobiMatrix'-object.
 */
class FurnaceIngotEqn : public ODEObject {
public:
	//constructor
	FurnaceIngotEqn (continuous::Continuous* ingot)
	:	ingot(ingot)
	,	ingotState(ingot)
	,	dfdt(ingot)
	{}

	void derivates(SimTime time) {
		if (continuous == 0) {
			throw new NotAssignedException("Continous", "FurnaceIngotEqn");
		}
		// the continuous object has to be set here, because the continuous pointer is set to 0 before.
		// 'continuous'- object equates to the 'furnace'-object.
		rate.setContinuous(continuous);
		state.setContinuous(continuous);

		// u'(t) = (500 - u(t))/n - (u(t) - x(t))/3
		rate[0] = (500 - state[0])/inFurnace - (state[0]-ingotState[0])/3;
	}

	void jacobi(SimTime time) {
		if (continuous == 0 || ingot == 0) {
			throw new NotAssignedException("Continous", "FurnaceIngotEqn");
		}

		// the continuous object has to be set here, because the continuous pointer is set to 0 before.
		// 'continuous'- object equates to the 'furnace'-object.
		ingotJacobiMatrix.setContinuous(continuous, ingot);

		// Df/Du = 0
		// Df/Dx = 1/3
		ingotJacobiMatrix(0,0) = 1.0 / 3.0;

		//DfDt is zero as system is autonomic
		dfdt[0] = 0;
	}

private:
	continuous::Continuous* ingot;
	Rate rate;
	State state, ingotState;
	JacobiMatrix ingotJacobiMatrix;
	DfDt dfdt;
};

/** Tests if an ingot is hot enough to leave the furnace, that means if the temperature of the ingot reached 380 C the ingot leaves the furnace.
 */
class HeatReached : public StateEvent {
public:
	//constructor
	HeatReached (continuous::Continuous* ingot, continuous::Continuous* furnace, FurnaceEqn* furnaceEqn, FurnaceIngotEqn* eqn1, IngotEqn* eqn2)
	:	ingot(ingot)
	,	furnace(furnace)
	,	furnaceEq(furnaceEqn)
	,	furnaceIngotEqn(eqn1)
	,	ingotEqn(eqn2)
	{}

	// If the temperature of the ingot reaches 380 C ...
	bool condition(SimTime time) {
		return ingot->getValue(0) >= 380;
	}

	// ... then set the 'ingotEqn'-object to the state 'in the air'
	// and remove the 'furnaceIngotEqn'-object from the furnace object.
	// If there is no ingot in the furnace, add the 'furnaceEqn'-object to the 'furnace'-object (the furnace is now empty again).
	void action() {
		ingotEqn->inFurnace(false);
		furnace->removeODE(furnaceIngotEqn);
		if(--inFurnace==0){
			furnace->addODE(furnaceEq);
		}
	}

private:
	continuous::Continuous* ingot, *furnace;
	FurnaceEqn* furnaceEq;
	FurnaceIngotEqn* furnaceIngotEqn;
	IngotEqn* ingotEqn;
};

/** Tests if an ingot has to be transported into the furnace.
 */
class EnterFurnace : public StateEvent {
public:
	//constructor
	EnterFurnace (continuous::Continuous* furnace, FurnaceEqn* furnaceEqn, FurnaceIngotEqn* eqn1,
			IngotEqn* eqn2, SimTime enterTime)
	:	furnace(furnace)
	,	furnaceIngotEqn(eqn1)
	,	furnaceEqn(furnaceEqn)
	,	ingotEqn(eqn2)
	,	enterTime(enterTime)
	{}

	// If the time when an ingot has to be transported into the furnace (the times are: 0, 3, 6, 9) is reached ...
	bool condition(SimTime time) {
		if(time >= enterTime) {
			return true;
		}
		return false;
	}

	// ... then set the 'ingotEqn'-object to the state 'inside the furnace'.
	// Because of the furnace is not empty, remove the equation for an empty furnace and add
	// the furnace/ingot equation to the furnace object.
	// Also increase the inFurnace counter.
	void action() {
		ingotEqn->inFurnace(true);
		if(inFurnace==0){
			furnace->removeODE(furnaceEqn);
		}
		furnace->addODE(furnaceIngotEqn);
		++inFurnace;
	}

private:
	continuous::Continuous* furnace;
	Monitor* monitor;
	FurnaceIngotEqn* furnaceIngotEqn;
	FurnaceEqn* furnaceEqn;
	IngotEqn* ingotEqn;
	SimTime enterTime;
};

//
// The process describes the discrete part of the lifecycle of one ingot.
// It adds the 'in the air' ingot equation to a continuous ingot object.
// It also adds the discrete state events for entering and leaving
// the furnace to the monitor.
//
class Ingot : public Process {
public:
	//constructor
	Ingot(Simulation& sim, continuous::Continuous* ingot, IngotEqn* ingotEqn, Monitor* monitor, HeatReached* heatReached, EnterFurnace* enterFurnace)
	:	Process(sim, "Ingot_Process")
	,	ingotEqn(ingotEqn)
	,	ingot(ingot)
	,	monitor(monitor)
	,	heatReached(heatReached)
	,	enterFurnace(enterFurnace)
	{}
	//main
	int main() {
		ingot->addODE(ingotEqn);
		monitor->addStateEvent(enterFurnace);
		monitor->addStateEvent(heatReached);
		return 0;
	}


private:
	IngotEqn* ingotEqn;
	continuous::Continuous* ingot;
	Monitor* monitor;
	HeatReached* heatReached;
	EnterFurnace* enterFurnace;
};

int main() {

	// get simulation
	Simulation& sim = odemx::getDefaultSimulation();

	//enable logging
	//sim.enableDefaultLogging( odemx::data::output::STDOUT );

	//init monitor and solver
	GSLSolver* solver = new GSLSolver(sim, -3);
	solver->setErrorType(absolute);
	solver->setErrorLimit(0.1);
	solver->setStepLength(0.01, 0.1);

	// The solver is initialized by the monitor, by calling the method solver->init(), to get it ready to work, before the first integration step.
	// Also the monitor calls the method setODESolver and during this call the solver returns a suitable
	// variable container for holding all variables needed.
	Monitor* monitor = new Monitor(sim, "Furnace-Ingot-Example", 5, solver);

	// init continuous objects
	// Below follow the continuous objects which contain the equations for the ingots and for the furnace.
	// They also are the access points to the values needed by the equation objects.
	continuous::Continuous *furnace = new continuous::Continuous(sim, "Furnace", monitor, 1),
	*ingot1 = new continuous::Continuous(sim, "Ingot1", monitor, 1),
	*ingot2 = new continuous::Continuous(sim, "Ingot2", monitor, 1),
	*ingot3 = new continuous::Continuous(sim, "Ingot3", monitor, 1),
	*ingot4 = new continuous::Continuous(sim, "Ingot4", monitor, 1);

	// equation modifiers for furnace by ingots
	// These objects describe the state changes of the furnace temperature
	// while one or more ingots are in the furnace.
	FurnaceIngotEqn* furnaceIngot1Eqn = new FurnaceIngotEqn(ingot1),
	*furnaceIngot2Eqn = new FurnaceIngotEqn(ingot2),
	*furnaceIngot3Eqn = new FurnaceIngotEqn(ingot3),
	*furnaceIngot4Eqn = new FurnaceIngotEqn(ingot4);

	// These objects describe the state changes of the ingots temperature while they are inside or outside the furnace.
	IngotEqn *ingot1Eqn = new IngotEqn(ingot1, furnace)
	, *ingot2Eqn = new IngotEqn(ingot2, furnace)
	, *ingot3Eqn = new IngotEqn(ingot3, furnace)
	, *ingot4Eqn = new IngotEqn(ingot4, furnace);

	// This equation object describes the furnace temperature while it is empty.
	FurnaceEqn* furnaceEqn = new FurnaceEqn(furnace);

	// This equation is added to the 'furnace'-object, because the furnace is empty initially.
	furnace->addODE(furnaceEqn);

	// State events to check when ingot has to leave furnace, that means if the temperature of an ingot reaches 380 C.
	HeatReached *heatIngot1 = new HeatReached(ingot1, furnace, furnaceEqn, furnaceIngot1Eqn, ingot1Eqn),
	*heatIngot2 = new HeatReached(ingot2, furnace, furnaceEqn, furnaceIngot2Eqn, ingot2Eqn),
	*heatIngot3 = new HeatReached(ingot3, furnace, furnaceEqn, furnaceIngot3Eqn, ingot3Eqn),
	*heatIngot4 = new HeatReached(ingot4, furnace, furnaceEqn, furnaceIngot4Eqn, ingot4Eqn);

	// State events to check if the time  is reached for the ingot to enter furnace.
	EnterFurnace *Enter1 = new EnterFurnace(furnace, furnaceEqn, furnaceIngot1Eqn, ingot1Eqn, 12.0),
	*Enter2 = new EnterFurnace(furnace, furnaceEqn, furnaceIngot2Eqn, ingot2Eqn, 15.0),
	*Enter3 = new EnterFurnace(furnace, furnaceEqn, furnaceIngot3Eqn, ingot3Eqn, 18.0),
	*Enter4 = new EnterFurnace(furnace, furnaceEqn, furnaceIngot4Eqn, ingot4Eqn, 21.0);

	// Discrete processes to activate the ingots in the simulation.
	Ingot *ingotProcess1 = new Ingot(sim, ingot1, ingot1Eqn, monitor, heatIngot1, Enter1),
	*ingotProcess2 = new Ingot(sim, ingot2, ingot2Eqn, monitor, heatIngot2, Enter2),
	*ingotProcess3 = new Ingot(sim, ingot3, ingot3Eqn, monitor, heatIngot3, Enter3),
	*ingotProcess4 = new Ingot(sim, ingot4, ingot4Eqn, monitor, heatIngot4, Enter4);

	// Initialization of the continuous system, by setting the initial values.
	furnace->setValue(0, 400);
	ingot1->setValue(0, 150);
	ingot2->setValue(0, 150);
	ingot3->setValue(0, 150);
	ingot4->setValue(0, 150);

	// Initialization of the monitor, by activating the monitor, setting the time limit and adding the corresponding processes to the waiting list.
	monitor->activate();
	monitor->setTimeLimit(100.0);
	monitor->setInternalTime(0.0);

	monitor->addProcessToWaitList(ingotProcess1);
	monitor->addProcessToWaitList(ingotProcess2);
	monitor->addProcessToWaitList(ingotProcess3);
	monitor->addProcessToWaitList(ingotProcess4);

	// Initialization of the continuous observers.
	ContinuousTrace furnaceTrace(furnace), ingot1Trace(ingot1), ingot2Trace(ingot2), ingot3Trace(ingot3), ingot4Trace(ingot4);

	// Activation of all other discrete ingot processes.
	ingotProcess1->activateAt(2.0);
	ingotProcess2->activateAt(3.0);
	ingotProcess3->activateAt(6.0);
	ingotProcess4->activateAt(9.0);

	// Start simulation
	sim.runUntil(100);
	return 0;
}

#else // ODEMX_USE_CONTINUOUS not defined

#include <iostream>

int main( int argc, char* argv[] )
{
	std::cerr << "ODEMx compiled without support for continuous processes."
			  << std::endl;
	return 0;
}

#endif /* ODEMX_USE_CONTINUOUS */
