//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Example_Wave.cpp

 \author Sascha Qualitz
 <!-- [\author <author>]* -->

 \date created at 2010/12/21

 <!-- [detailed description] -->

 <!-- [\todo {todos for this file}]* -->

 \since 3.0
 */
#include <odemx/odemx.h>

#include <stdio.h>
#include <time.h>
#include <typeinfo>
#include <cmath>

/** \example Example_Wave.cpp

	This example is to show cellular automaton simulation techniques of ODEMx.
	A cellular automaton is a 4-tuple $(L,S,N,f)$, where L is a regular grid,
	S a finite set of states, N a finite set of neighborhood indices and f a
	function for the calculation of the state transitions. A cellular automaton
	is time driven, so the state changes from time t to t+1.

	This example describes a simulation of a wave beginning in the middle of
	a two dimensional 20x20-grid and then expands to the boundary of the grid.
	Each cell has three states, numerical 0,1,2. Where 0 equates to a resting state,
	1 equates to a recovering state an 2 equates to exciting state.
*/

using namespace odemx::base;
using namespace odemx::base::cellular;
using namespace odemx;
using namespace std;

/** The transition function

	Any cell object in the grid owns one object of type WaveTransition. In this class you
	have to implement the two methods void transitionFunction(SimTime time) and
	void outputFunction(). In this example the rules for the state transition are implemented
	in the first method. In the second method the output function of the automaton is implemented.

	There are three rules for the computation of the state changes written in comments
	following in the code. Before the first rule is checked in the code, there is checked
	if any neighbors have the state 2.
*/
class WaveTransition : public Transition {
public:
	void transitionFunction(SimTime time) {
		std::vector<double> vec = this->cell->pullValue(0);
		unsigned waveNeighbor = 0;

		for(int i = 0; i < vec.size(); ++i){
			if(vec[i] == 2) {
				waveNeighbor++;
			}
		}

		/*	If the cell is in state 1, the state of the cell will be 0 and the cell is able
			to change the state to 2 if rule one fits.*/
		if (getValue(0) == 1) {
			setValue(0,0);
			return;
		}

		/*	If the cell is in state 2, the state of the cell will be 1.*/
		if (getValue(0) == 2) {
			setValue(0,1);
			return;
		}
		/*	If the cell is in state 0 and there are one or more cells in the neighborhood
			with state 2, then the state of the cell will be 2 too.*/
		if (waveNeighbor != 0) {
			if (getValue(0) == 0) {
				setValue(0,2);
				return;
			}

		}

	}
	void outputFunction() {
		cell->setOutputValue(0,cell->getValue(0));
	}

};

//---------------------------------------------------------------------------------------------PrinterProcess
/** This class visualizes the 2d-grid and the changing state of the cells.
	But the state 0 equates here to a blank position. So only the states 1 and 2
	are shown.
 */
class Visualizer : public Process {
public:
	//constructor
	Visualizer(Simulation& sim, SimTime startTime, SimTime stepSize, SimTime endTime, CellMonitor *monitor)
	: Process(sim, "Configuration of the ca")
	, startTime(startTime)
	, stepTime(stepSize)
	, endTime(endTime)
	, monitor(monitor)
	{}
	//main
	int main() {

		do {
			system("clear");
			printf("      +");
			for (int i = 0; i < this->monitor->getColumnCount()+2; i++) {printf(" ");}
			printf(" +\n");
			printf("	");
			printf("Time: %d \n", (int)(this->monitor->getTime()) );
			printf("	");
			for (int i = 0; i < this->monitor->getColumnCount()+2; i++) {printf("-");}
			printf("\n");
			printf("	");
			for (int k = 0; k < this->monitor->getLevelCount(); k++) {
				for (int i = 0; i < this->monitor->getRowCount(); i++) {
					printf("|");
					for (int j = 0; j < this->monitor->getColumnCount(); j++) {
						if(this->monitor->getCell(i,j,k)->isPartOfBoundary()) {
							printf(" ");
							continue;
						}
						if(monitor->getCell(i,j,k)->getValue(0) == 0) {
							printf(" ");
						} else {
							printf("%.0f", monitor->getCell(i,j,k)->getValue(0));
						}
						if( j != 0 && (j+1) % this->monitor->getColumnCount() == 0 ) {
							printf("|\n");
							printf("	");
						}
						if( i*j != 0 && (i+1)*(j+1) % (this->monitor->getRowCount()*this->monitor->getColumnCount()) == 0 ) {
							for (int i = 0; i < this->monitor->getColumnCount()+2; i++) {printf("-");}
							printf("\n");
							printf("      +");
							for (int i = 0; i < this->monitor->getColumnCount()+2; i++) {printf(" ");}
							printf(" +\n");
						}
					}
				}
			}
			activateIn(stepTime);
		} while (getSimulation().getTime() <= endTime);
	
		printf("# printing done.\n");
		return 0;
}


	
private:
	SimTime startTime, stepTime, endTime;
	std::list<CellMonitor*> watchedValues;
	CellMonitor *monitor;
	double* array;
};

int main() {
	// get simulation
	base::Simulation& sim = getDefaultSimulation();

	//enable logging
	//sim.enableDefaultLogging( odemx::data::output::STDOUT );

	// The monitor is initialized with an object of type Simulation, its label
	// and the path to the configuration file and the file which contains the
	// initial state value of the cell at the position (10,10,0), where format
	// is (row index, column index, level index).
	CellMonitor* cellMonitor = new CellMonitor(sim,"Example_Wave","Example_Wave_config.xml","Example_Wave_ca_start_data.xml");

	// Initialization of the monitor observer, which generates a trace file with
	// the name Example_Wave_trace.xml.
	CellMonitorTrace cellMonitorTrace(cellMonitor);

	// This macro registers objects of type WaveTansition at every cell object.
	// It uses the three macros ROWSPAN, COLUMNSPAN, and LEVELSPAN to encapsulate
	// the span indices.
	_setTransitionToCells(ROWSPAN(0,20), COLUMNSPAN(0,20), LEVELSPAN(0,0), cellMonitor, WaveTransition);

	// The visualizer object to print the state changes in every step.
	Visualizer *visualizer = new Visualizer(sim, 0, 1, 12, cellMonitor);

	// Initialization of the monitor, by activating the monitor and setting the time limit
	// and also activates the visualizer.
	cellMonitor->activate();
	cellMonitor->setTimeLimit(12);
	visualizer->activate();

	// Start simulation
	sim.runUntil(12);
	sim.exitSimulation();

	return 0;
}
