//------------------------------------------------------------------------------
//	Copyright (C) 2009 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/**
 * @file Example_Consumer.cpp
 * @author Ronald Kluth
 * @date created at 2010/03/16
 * @brief Example displaying the usage of ODEMx text logging components
 * @since 3.0
 */

/**
 * @example Example_Consumer.cpp
 * The usage of text logging ODEMx components is demonstrated.
 *
 * First, a time base is created with start time and time unit.
 * This time base is used to initialize two time formatters, one for
 * German time, and the other for the ISO 8601 format.
 *
 * These time formats are the used for two log output components,
 * one that writes to console, and another one that creates XML files.
 * Finally, a data Producer object is used to create some log records
 * for display.
 */
#include "Example_Producer_DebugStopWatch.h"
#include <iostream>
using namespace odemx::data::output;

int main() {
	//
	// We create a few typedef to make the code more readable.
	//
	typedef std::tr1::shared_ptr< OStreamWriter > OStreamWriterPtr;
	typedef std::tr1::shared_ptr< XmlWriter > XmlWriterPtr;
	//
	// ODEMx provides the capability to format simulation time values
	// as real dates and daytime. All that is needed is a starting point
	// in time and a time unit. The TimeFormat subclasses can then compute
	// the relative time values and transform them into strings.
	//
	// The time base we choose here is December 30th, 2009. The daytime
	// offset is at 10:00 o'clock.
	//
	TimeBase startTime( 2009, 12, 30, 10*60*60*1000, TimeUnit::milliseconds );
	//
	// Next, we create two test-writing components. The OStreamWriter can
	// be used to write log data to any std::ostream. A typical use case
	// is the console output.
	//
	// The XmlWriter on the other hand creates XML files and fills them
	// up to the given limit of 1000 records.
	//
	OStreamWriterPtr os = OStreamWriter::create( std::cout );
	os->setTimeFormat( new GermanTime( startTime ) );
	XmlWriterPtr xml = XmlWriter::create( "Example_Consumer", 1000 );
	xml->setTimeFormat( new Iso8601Time( startTime, 1, 0 ) );
	//
	// Since the Simulation context is the LoggingManager in ODEMx,
	// we can simply call the method addConsumer to register the
	// output components with the log channels debug and info.
	//
	odemx::base::Simulation& sim = odemx::getDefaultSimulation();
	sim.addConsumer( odemx::data::channel_id::debug, os );
	sim.addConsumer( odemx::data::channel_id::info, os );
	sim.addConsumer( odemx::data::channel_id::info, xml );
	//
	// Finally, we start a log data producer to generate some output.
	//
	DebugStopWatch watch( sim, "Stop Watch" );
	watch.start();
	sim.setCurrentTime( 42 );
	watch.stop();
}
