/*****************************************************************************
  Copyright (C) 1993, 2001, 2011 Humboldt-Universitaet zu Berlin

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*****************************************************************************/

/**
 * @file   Example_CA_GameOfLife.cpp
 * @author Oliver Keil
 * @date   created at 2011/06/09
 * @brief  Simple Cellular Automaton example, Game-Of-Life.
 * @since  3.0
 **/

/**
 * @example Example_CA_GameOfLife.cpp
 * Shows basic usage of Cellular Automaton implementation.
 *
 * This demonstrates how to use Cellular Automata in ODEMx.  For this a simple
 * Game-Of-Life automaton is implemented.
 *
 * We use a 'Cell'-derivation named 'GameOfLifeCell' to define cells of our
 * automaton.  To describe a cell's behavior we need to override
 * 'transitionFunction'.  So, our 'transitionFunction' implements the well
 * known Game-Of-Life rules.
 *
 * Functions 'getMooreNeighbors', 'getValue', and 'setValue' are defined to
 * ease the implementation of 'transitionFunction'.
 *
 * In 'main' we create a new 'CellMonitor' using our new defined
 * 'GameOfLifeCell'.  Before starting this new automaton it gets initialized
 * with a simple "glider".  Finally, the automaton runs for some time and
 * program will be closed explicitly.
 **/
#include <odemx/odemx.h>

#include <string>
#include <iostream>

using namespace odemx::base::cellular;

class GameOfLifeCellTransition : public Transition {
public:
	void transitionFunction(SimTime time) {

		using std::cout;
		using std::endl;

		unsigned int livingNeighbors = 0;

		// estimate living neighbor cells
		std::vector<double> neighborValues = this->cell->pullValue(0);
		for (unsigned int i = 0; i < neighborValues.size(); ++i)
			if (neighborValues[i] == 1) ++livingNeighbors;

		// change cell's state according to well known Game-Of-Life rules
		if (getValue() == 0 && livingNeighbors == 3) {
			setValue (0,1);
		}
		if (getValue() == 1 && (livingNeighbors < 2 || livingNeighbors > 3)) {
			setValue (0,0);
		}

	}
	void outputFunction() {
		cell->setOutputValue(0,cell->getValue(0));
	}

};


//
// Helper function to print a 2-dimensional cellular automaton.
// Living cells are marked by '*' and dead ones by '.'.
//
void printCells (odemx::base::cellular::CellMonitor* cm) {
	unsigned int rows = cm->getRowCount();
	unsigned int cols = cm->getColumnCount();
	double* valuesVector = cm->getValues();
	using std::cout;
	using std::endl;

	for (unsigned int index = 0; index < (rows * cols); index++) {
		cout << (valuesVector[index] ? "*" : ".") << " ";
		if (! ((index + 1) % cols) )
			cout << endl;
	}
	cout << endl << endl;
}

//
// Helper function to set a glider on position (x,y).  Position marks
// upper-left corner of glider. The upper-left corner of whole 2-dimensional
// cellular automaton equals position (0,0).
//
// Fails if glider is positioned wrongly.
//
int setGlider (odemx::base::cellular::CellMonitor* cm, unsigned int posx
		, unsigned int posy)
{
	unsigned int rows = cm->getRowCount();
	unsigned int cols = cm->getColumnCount();

	// glider needs space of 3x3 cells to be set correctly
	if ((posx + 2) > cols)
		return -1;
	if ((posy + 2) > rows)
		return -1;
		
	using std::cout;
	using std::endl;

	//   *
	//     *
	// * * *
	cm->getCell((posx + (cols *  posy)     ) + 1)->setValue(0,1);	
	cm->getCell((posx + (cols * (posy + 1))) + 2)->setValue(0,1);
	cm->getCell((posx + (cols * (posy + 2)))    )->setValue(0,1);
	cm->getCell((posx + (cols * (posy + 2))) + 1)->setValue(0,1);
	cm->getCell((posx + (cols * (posy + 2))) + 2)->setValue(0,1);
	
	// set the first value that is seen by other cells
	cm->getCell((posx + (cols *  posy)     ) + 1)->setOutputValue(0,1);
	cm->getCell((posx + (cols * (posy + 1))) + 2)->setOutputValue(0,1);
	cm->getCell((posx + (cols * (posy + 2)))    )->setOutputValue(0,1);
	cm->getCell((posx + (cols * (posy + 2))) + 1)->setOutputValue(0,1);
	cm->getCell((posx + (cols * (posy + 2))) + 2)->setOutputValue(0,1);

	return 0;
}


int main () {
	odemx::base::Simulation& sim = odemx::getDefaultSimulation();

	//unsigned int cellDimension = 1;
	//GameOfLifeCell cell (cellDimension);

	odemx::data::Label monitorName = "Game of Life Monitor";
	unsigned int inCount = 1;        // number of inputs per cell
	unsigned int outCount = 1;       // number of outputs per cell
	unsigned int cols = 10;          // columns and rows and level of automaton
	unsigned int rows = 10;          //
	unsigned int level = 1;			 //
	unsigned int cellDimension = 1;  // dimesnion of the cells
	double       timestep = 1;
	odemx::base::cellular::CellMonitor *cm = new odemx::base::cellular::CellMonitor(sim, monitorName, inCount, outCount, rows, cols, level, timestep, cellDimension);
	
	// This macro registers objects of type CellTransition at every cell object.
	// It uses the three macros ROWSPAN, COLUMNSPAN, and LEVELSPAN to encapsulate
	// the span indices.
	_setTransitionToCells(ROWSPAN(0,9), COLUMNSPAN(0,9), LEVELSPAN(0,0), cm, GameOfLifeCellTransition);

	setGlider (cm, 0, 0);

	cm->setTimeLimit (45.0);
	cm->activate();

	printCells (cm);
	for (int i = 0; i < 45; i++) {
		sim.step();
		printCells (cm);
	}

	return 0;
}
