/**
 * @file TestRandomInt.cpp
 * @date Aug 9, 2008
 * @author Ronald Kluth
 *
 * @brief Tests for ODEMx class RandomInt
 */

#include "TestRandom.h"
#include "../TestBase/TestBase.h"
#include "../TestData/TestData.h"
#include <cstdlib> // atof()

/// @cond DOXYGEN_SKIP
SUITE( Random )
{
/// @endcond

	/**
	 * @struct RandomIntFixture
	 * @brief Helper struct providing set-up/tear-down of RandomInt tests
	 *
	 * @copydetails EventFixture
	 */
	struct RandomIntFixture
	{
		SuiteBase::SimulationTest sim;
		TestLogConsumer::Ptr log;
		int lower;
		int upper;
		int cells;
		RandomInt rand;
		SuiteData::ReportTest reporter;
		statistics::Histogram histogram;
		data::TypeInfo type;

		RandomIntFixture()
		:	sim( "RandomIntTestSim" ),
			log( TestLogConsumer::create() ),
			lower( 0 ),
			upper( 4 ),
			cells( upper - lower ),
			rand( sim, "RandomIntTest", lower, upper ),
			histogram( sim, "RandomIntTestHistogram", lower, upper, cells ),
			type( typeid(RandomInt) )
			{
				sim.addConsumer( log );
			}
	};

	/**
	 * @test odemx::RandomInt construction
	 *
	 * Expected function call effects:
	 * @li lower bound greater upper bound produces a warning and switches
	 * the two values
	 * @li label is set
	 * @li dist context (simulation) is set
	 * @li lower and upper bounds are set
	 */
	TEST_FIXTURE( RandomIntFixture, ConstructionDestruction )
	{
		data::Label l = "RandomIntTestUserSimConstruction";
		{
			RandomInt rand2( sim, l, upper, lower );
			CHECK( log->getTraceRecord( "create", type ) );
			CHECK( log->getStatisticsRecord( "parameter", "seed", type ) );
			CHECK( log->getStatisticsRecord( "parameter", "lower bound", type ) );
			CHECK( log->getStatisticsRecord( "parameter", "upper bound", type ) );
			CHECK( log->getWarningRecord( "RandomInt(): lower bound greater than upper bound; bounds swapped", type ) );
			CHECK_EQUAL( l, rand2.getLabel() );
			CHECK_EQUAL( lower, rand2.getLowerBound() );
			CHECK_EQUAL( upper, rand2.getUpperBound() );
		}
		CHECK( log->getTraceRecord( "destroy", type ) );
	}

	/**
	 * @test odemx::RandomInt::sample()
	 *
	 * Expected function call effects:
	 * @li samples are always within the interval [lower, upper]
	 * @li all values have roughly the same hit percentage
	 */
	TEST_FIXTURE( RandomIntFixture, Sample )
	{
		rand.sample();
		CHECK( log->getTraceRecord( "sample", type ) );
		CHECK( log->getStatisticsRecord( "count", "uses", type ) );

		for( int i = 0; i < 1500000; ++i )
		{
			int value = rand.sample();
			CHECK( rand.getLowerBound() <= value );
			CHECK( rand.getUpperBound() >= value );
			histogram.update( value );
		}

		reporter.addReportProducer( histogram );
		reporter.generateReport();
		CHECK( reporter.processedTables );

		vector< string >& table = reporter.allTableContent[1];
		// actual cell count equals cells + 2 because of values
		// greater or less than histogram range
		CHECK_EQUAL( 0, atof( table[ 0 * 5 + 3 ].c_str() ) );
		CHECK_EQUAL( 0, atof( table[ (cells + 1) * 5 + 3 ].c_str() ) );
		for( int i = 1; i < cells + 1; ++i )
		{
			CHECK_CLOSE( 100.0 / (cells), atof( table[ i * 5 + 3 ].c_str() ), 0.1  );
		}

		// print histogram
//		for( size_t i = 0; i < table.size(); ++i )
//		{
//			cout << table[i] << " ";
//			if( ( i % 5 ) == 4 ) cout << endl;
//		}
//		cout << endl;
	}

/// @cond DOXYGEN_SKIP
}
/// @endcond
