/**
 * @file TestDistContext.cpp
 * @date Aug 9, 2008
 * @author Ronald Kluth
 *
 * @brief Tests for ODEMx class DistContext
 */

#include "TestRandom.h"
#include "../TestBase/TestBase.h"

/// @cond DOXYGEN_SKIP
SUITE( Random )
{
/// @endcond

	/**
	 * @struct DistContextFixture
	 * @brief Helper struct providing set-up/tear-down of DistContext tests
	 *
	 * @copydetails EventFixture
	 */
	struct DistContextFixture
	{
		SuiteBase::SimulationTest sim;

		DistContextFixture()
		:	sim( "DistContextTestSim" )
			{}
	};

	/**
	 * @test odemx::DistContext construction
	 *
	 * Expected function call effects:
	 * @li initial seed is set to a default value
	 */
	TEST_FIXTURE( DistContextFixture, ConstructionDestruction )
	{
		CHECK_EQUAL( (unsigned long) 907, sim.getSeed() );
	}

	/**
	 * @test odemx::DistContext::setSeed(int)
	 *
	 * Expected function call effects:
	 * @li allowed values are set as is
	 * @li a negative value is turned into a positive value
	 * @li a too large value is decreased by \c odemx::zyqmodulo
	 * @li a value of \c 0 will cause seed to be set to a default value
	 */
	TEST_FIXTURE( DistContextFixture, SetSeed )
	{
		unsigned long seed = 125;
		sim.setSeed( seed );
		CHECK_EQUAL( seed, sim.getSeed() );

		seed = 125;
		long negativeSeed = 0 - seed;
		sim.setSeed( negativeSeed );
		CHECK_EQUAL( seed, sim.getSeed() );

		seed = zyqmodulo + 25;
		sim.setSeed( seed );
		CHECK_EQUAL( seed - ( (int)(seed/zyqmodulo) )*zyqmodulo, sim.getSeed() );

		sim.setSeed( 0 );
		CHECK_EQUAL( (unsigned long)( zyqmodulo / 2 ), sim.getSeed() );
	}

	/**
	 * @test odemx::DistContext::getNextSeed()
	 *
	 * Expected function call effects:
	 * @li all sample values fall within the interval [0,odemx::zyqmodulo]
	 */
	TEST_FIXTURE( DistContextFixture, GetNextSeed )
	{
		for( int i = 0; i < 10000; ++i )
		{
			CHECK( 0 <= sim.getNextSeed() );
			CHECK( zyqmodulo >= sim.getNextSeed() );
		}
	}

/// @cond DOXYGEN_SKIP
}
/// @endcond
