/**
 * @file TestSimRecord.cpp
 * @date May 3, 2010
 * @author ron
 *
 * @brief
 */

#include "TestData.h"
#include "../TestBase/TestBase.h"

/// @cond DOXYGEN_SKIP
SUITE( Data )
{
/// @endcond

	/**
	 * @struct SimRecordFixture
	 * @brief Helper struct providing set-up/tear-down of SimRecord tests
	 *
	 * @copydetails EventFixture
	 */
	struct SimRecordFixture
	{
		SuiteBase::SimulationTest sim;
		TestLogConsumer::Ptr log;
		ProducerTest producer;
		TypeInfo type;

		SimRecordFixture()
		:	sim( "SimRecordTestSim" )
		,	log( TestLogConsumer::create() )
		,	producer( sim, "SimRecordTestFixtureProducer" )
		,	type( typeid(ProducerTest) )
		{
			sim.addConsumer( log );
		}
	};

	TEST_FIXTURE( SimRecordFixture, Construction )
	{
		base::SimTime time = 123;
		sim.setCurrentTime( time );
		StringLiteral msg = "SimRecordConstructionTest";
		SimRecord rec( sim, producer, msg );
		CHECK_EQUAL( msg, rec.getText() );
		CHECK_EQUAL( &sim, &rec.getSimulation() );
		CHECK_EQUAL( &producer, &rec.getSender() );
		CHECK_EQUAL( time, rec.getTime() );
	}

	TEST_FIXTURE( SimRecordFixture, Scope )
	{
		StringLiteral msg = "SimRecordScopeTest";
		SimRecord rec( sim, producer, msg );
		rec.scope( type );
		CHECK( rec.hasScope() );
		CHECK( type == rec.getScope() );
	}

	TEST_FIXTURE( SimRecordFixture, Detail )
	{
		StringLiteral msg = "SimRecordDetailTest";
		SimRecord rec( sim, producer, msg );

		CHECK_THROW( rec.getDetails(), odemx::DataException );

		rec.detail( "SimRecordTestDetail", 99 );
		CHECK( rec.hasDetails() );
		CHECK_EQUAL( StringLiteral( "SimRecordTestDetail" ), rec.getDetails().front().first );
		CHECK_EQUAL( 99, rec.getDetails().front().second.convert<int>() );
	}

	TEST_FIXTURE( SimRecordFixture, ValueChange )
	{
		StringLiteral msg = "SimRecordValueChangeTest";
		SimRecord rec( sim, producer, msg );
		rec.valueChange( 99, 999 );
		CHECK( rec.hasDetails() );
		CHECK_EQUAL( StringLiteral( "old value" ), rec.getDetails().front().first );
		CHECK_EQUAL( 99, rec.getDetails().front().second.convert<int>() );
		CHECK_EQUAL( StringLiteral( "new value" ), rec.getDetails().back().first );
		CHECK_EQUAL( 999, rec.getDetails().back().second.convert<int>() );
	}

/// @cond DOXYGEN_SKIP
}
/// @endcond
