//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file GSLSolver.h

 \author Michael Fiedler

 \date created at 2008/11/21

 \brief Declaration of GSLSolver

 \since 3.0
 */

#ifndef ODEMX_GSLSOLVER_INCLUDED
#define ODEMX_GSLSOLVER_INCLUDED

#include <odemx/setup.h>

// usage of this class requires SimTime type double,
// which can be switched at compile time by defining
// ODEMX_USE_CONTINUOUS in file odemx/setup.h
#ifdef ODEMX_USE_CONTINUOUS

#include <odemx/base/continuous/ODESolver.h>
#include <gsl/gsl_odeiv.h>

/** give function in GSL form global function
 */
int gsl_func(double t, const double y[], double dydt[], void* params);

/** give jacobi in GSL form global function
 */
int gsl_jacobian(double t, const double y[], double * dfdy, double dfdt[], void* params);

namespace odemx {
	namespace base {
		namespace continuous {

			//forward declaration
			class Monitor;

			class Continuous;

			class VariableContainer;

			class ODEObject;

			class ODESolver;

			/** \class GSLSolver

				\ingroup base

				\brief An Implementation of the abstract class ODESolver which uses the GNU scientific library

				\note GSLSolver supports Observation

				\note GSLSolver supports Trace

				\note GSLSolver supports Logging

				\sa ODESolver

				\since 3.0

			 */
			class GSLSolver : public ODESolver {
			public:
				/** \brief Construction
					\param order
						parameter to select the used numeric solving algorithm.

					explicit:

						0..2 Runge-Kutta 2

						3 Runge-Kutta 4

						4 Runge-Kutta-Fehlberg 4 5

						5 Runge-Kutta Cash-Karp 4 5

						6..8 Runge-Kutta Prince-Dormand 8 9

					implicit:

						-1 Runge-Kutta 2

						-2 Runge-Kutta 4

						-3 Bulirsch-Stoer

						-4 Gear method M=1

						-5 Gear method M=2

					\note For positive values of order the number to select a method is nearly the consistency order.
					This means the error in one step is proportional to (step size)^(consistency order)
					\note Explicit methods are faster, but have problems with stiff equations.
						Implicit methods can deal with stiff equations.
				 */
				GSLSolver(Simulation& sim, int order);

				/// destruction
				~GSLSolver();

				/** \brief Returns a variable container suitable for this Solver
					\param size
						initial size of arrays in the variable container
				 */
				VariableContainer* getVariableContainer(int size);

				/** \brief Makes one integration step for monitor
					\param time
						internal time of assigned monitor.

					\note \p time is the time from which to do the step.
				 */
				void makeStep(double time);

				/** \brief Initialize the solver

					\note This is also called if the monitor has changed.
				 */
				void init();

			protected:
				/* the following functions are redefined with implementation linked to
				   base class for friend declaration
				*/
				 //@{
				/** \brief Calls all derivates functions of all ODEObject objects assigned to Continuous objects which are managed by the monitor
					\param time
						internal time of assigned monitor.
				     \note This function evaluates all equations of all continuous processes as one function
				 	 	   as sum of partial functions at time internalTime + time.
				 	 \note This function is passed to the GSL Library for solving the ordinary differential equations.
				 */
				void evalF(double time);

				/** \brief Calls all jacobian functions of all ODEObject objects assigned to Continuous objects which are managed by the monitor
					\param time
						internal time of assigned monitor.
				 	\note This function evaluates all jacobian functions of all continuous processes as one matrix
				 	 	  as sum of partial matrices at time internalTime + time. Also it evaluates the function for differentiating
						  the i-th component of f by time
					\note This function is passed to the GSL Library for solving the ordinary differential equations, if a solving algorithm for
						  stiff equations is used.
				 */
				void evalJacobian(double time);
				//@}

				/** \brief Returns the variables container of the assigned Monitor
				 */
				VariableContainer* getVariableContainerOfMonitor();

			private:
				double nextStep; //for evolving function

				// solving algorithm
				const gsl_odeiv_step_type * T;

				//step object
				gsl_odeiv_step * step;

				// error control object for step size adaption
				gsl_odeiv_control * control;

				// evolve object for stepwise evolution of solution
				gsl_odeiv_evolve * evolve;

				//as of type mismatch otherwise, these functions were defined as global functions
				//give function in GSL form global function
				friend int ::gsl_func(double t, const double y[], double dydt[], void* params);

				// give jacobi in GSL form global function
				friend int ::gsl_jacobian(double t, const double y[], double * dfdy, double dfdt[], void* params);
			};
		}
	}
}

#endif /* ODEMX_USE_CONTINUOUS */

#endif /* ODEMX_GSLSOLVER_INCLUDED */

