//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file GSLContainer.h

 \author Michael Fiedler

 \date created at 2008/11/21

 \brief Declaration of GSLContainer

 \since 3.0
 */

#ifndef ODEMX_GSLCONTAINER_INCLUDED
#define ODEMX_GSLCONTAINER_INCLUDED

#include <odemx/setup.h>

// usage of this class requires SimTime type double,
// which can be switched at compile time by defining
// ODEMX_USE_CONTINUOUS in file odemx/setup.h
#ifdef ODEMX_USE_CONTINUOUS

#include <odemx/base/continuous/ODESolver.h>
#include <odemx/data/Producer.h>
#include <odemx/data/Observable.h>

namespace odemx {
	namespace base {
		namespace continuous {

			//forward declarations
			//class Monitor; //really useful? Never used!

			class Continuous;

			class VariableContainer;

			class ODEObject;

			class ODESolver;

			class GSLContainerObserver;

			/** \class GSLContainer

				\ingroup base

				\brief Implementation of VariableContainer for GNU Scientific Library

				\sa VariableContainer

				\since 3.0
			 */
			class GSLContainer : public VariableContainer, public data::Producer, public data::Observable<GSLContainerObserver> {
			public:
				/** \brief Constructor
				 	\param sim
				 		reference to one simulation-object
					\param size
						initial size of Container
				 */
				GSLContainer(Simulation& sim, int size);

				/// Destructor
				~GSLContainer();

				/** \brief Copies the first container size values of the array \p y to the array \p values

					\note y has to have at least container size or the result is undefined
				 */
				void setValues(const double y[]);

				/** \brief Exchanges internal array for derivation of \p f

					\note All further changes will be done on \p f.
				 */
				void setDerivativesPointer(double f[]);

				/** \brief Exchanges the pointers of jacobian and dfdt, so that the pointers point to the destination of \p dfdy and \p dfdt

					\note All further changes will be done on \p dfdy and \p dfdt.
				 */
				void setJacobiPointer(double * dfdy, double dfdt[]);

				/** \brief Adapts the numbers of variables which can be stored in the container
					\param newSize
						new size of container

					\note Old content is preserved as long as it fits in \p newSize, everything over will be truncated.
				 */
				virtual void adaptSizeTo(int newSize);

				/** \name Variables

				 Variables are for holding values, while derivatives, jacobi and DfDt are for calculating.

				 @{
				 */
				/** \brief Sets all variables to 0.0
				 */
				virtual void nullVariables();

				/** \brief Sets value of one variable
				 \param num
				 	 index of variable on which \p value is set
				 */
				virtual void setVariable(int num, double value);

				/** \brief Returns the value of one variable
				 \param num
				 	 index of returned variable
				 */
				virtual double getVariable(int num);
				//@}

				/** \name temporary values

				 These values are used during the calculation of the intermediate steps.

				 @{
				 */
				/** \brief Sets all temporary variables to 0.0
				 */
				virtual void nullValues();

				/** \brief Copies values of temporary variables to the corresponding variables
				 */
				virtual void copyVariablesToValues();

				/** \brief Copies values of variables to the corresponding temporary variables
				 */
				virtual void copyValuesToVariables();

				/** \brief Sets value of one temporary variable
				 \param num
				 	 index of variable on which \p value is set
				 */
				virtual void setValue(int num, double value);

				/** \brief Returns value of one temporary variable
				 \param num
				 	 index of returned variable
				 */
				virtual double getValue(int num);
				//@}

				/** \name derivatives

				 @{
				 */
				/** \brief Sets all values in the derivatives-array to 0.0
				 */
				virtual void nullDerivatives();

				/** \brief Sets one value of the derivatives-array, where num is the index
				 \param num
				 	 index of variable on which \p value is set
				 */
				virtual void setDerivative(int num, double value);

				/** \brief Adds one value to one derivatives-array variable, whereas num is the index
				 \param num
				 	 index of variable to which \p value will be added
				 */
				virtual void addToDerivative(int num, double value);

				/** \brief Returns value from the derivatives-array at the index num
				 \param num
				 	 index of returned variable
				 */
				virtual double getDerivative(int num);
				//@}

				/** \name jacobi

				 @{
				 */
				/** \brief Sets all jacobian values to 0.0
				 */
				virtual void nullJacobi();

				/** \brief Sets value of one jacobian variable

				 	\note Sets variable(\p i ,\p j ) to \p value.
				 */
				virtual void setJacobi(int i, int j, double value);

				/** \brief Adds one value to one jacobian variable

				 	\note Adds \p value to variable(\p i ,\p j ).
				 */
				virtual void addToJacobi(int i, int j, double value);

				/** \brief Returns value of one jacobian variable

				 	\return variable(\p i ,\p j )
				 */
				virtual double getJacobi(int i, int j);
				//@}

				/** \name DfDt

				 	right hand side derived by time

				 @{
				 */
				/** \brief Sets all derived components to 0.0
				 */
				virtual void nullDfDt();

				/** \brief Sets value of one derived component
				 	\param num
				 	 	 index of derived component on which \p value is set
				 */
				virtual void setDfDt(int num, double value);

				/** \brief Adds one value to one derived component
				 	\param num
				 	 	 index of derived component to which \p value will be added
				 */
				virtual void addToDfDt(int num, double value);

				/** \brief get value of one derived component
				 	\param num
				 	 	 index of returned derived component
				 */
				virtual double getDfDt(int num);
				//@}

			private:
				int size;
				double* values;
				double* variables;
				double* derivates;
				double* jacobi;
				double* dfdt;

				//saves if pointer was set to an array from external
				bool derivatesSet;
				bool jacobiSet;

				friend class GSLSolver;
			};

			/** \interface GSLContainerObserver

				\author Sascha Qualitz

				\brief Observer for GSLContainer specific events

				\sa VariableContainer

				\since 3.0
			*/
			class GSLContainerObserver:
				public ProcessObserver
			{
			public:
				virtual void onCreate( GSLContainer* sender ) {} ///< Construction

				virtual void onAdaptSizeTo( GSLContainer* sender, int newSize ) {} ///< increasing the size of the arrays in the GSLContainer

				virtual void onSetDerivativesPointer( GSLContainer* sender) {} ///< sets pointer to new array

				virtual void onSetJacobiPointer( GSLContainer* sender) {} ///< sets pointers to new arrays
			};
		}
	}
}
#endif /* ODEMX_USE_CONTINUOUS */

#endif /* ODEMX_GSLCONTAINER_INCLUDED */
