//------------------------------------------------------------------------------
//	Copyright (C) 2009, 2010 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/**
 * @file TypeDefs.h
 * @author Ronald Kluth
 * @date created at 2009/02/13
 * @brief Declaration of types SchedList, ProcessList, Priority, SimulationId
 * @since 3.0
 */

#ifndef ODEMX_BASE_TYPEDEFS_INCLUDED
#define ODEMX_BASE_TYPEDEFS_INCLUDED

#include <Poco/UUID.h>

#include <list>

//----------------------------------------------------------forward declarations

namespace odemx {
namespace base {

class Process;
class Sched;

//-----------------------------------------------------------header declarations

/// List type to hold scheduled objects in order
typedef std::list< Sched* > SchedList;

/// List type to store process objects
typedef std::list< Process* > ProcessList;

/// ID type to provide simulation objects with unique IDs
typedef Poco::UUID SimulationId;

/**
 * @brief Numeric type to express scheduling and queueing priority
 *
 * The type Priority is used to set a priority for processes and events, which
 * allows users to influence scheduling or queueing order.
 */
typedef double Priority;

/**
	\brief Member-pointer type for coding conditions

	This pointer to member-function type is used for coding
	conditions. The condition-function should return true
	if the condition is fulfilled.
*/
typedef bool( Process::*Condition )();

/**
	\brief Member-pointer type for coding selections
	\param partner
		pointer to process to select

	This pointer to member-function type is used for coding
	selections. The condition-function should return true
	if the \p partner process fits the selection.
*/
typedef bool( Process::*Selection )( Process* partner );

/**
	\brief Member-pointer type for coding Process weight functions
	\param partner
		pointer to process to weight

	This pointer to member-function type is used for coding
	a weight function to determine the importance of the partner process.
	The weight function should return a higher number for a higher degree
	of importance. This type of function is used in the implementation of
	WaitQ to synchronize processes according to their highest weight in
	relation to each other.
*/
typedef double( Process::*Weight )( Process* partner );

} } // namespace odemx::base

#endif /* ODEMX_BASE_TYPEDEFS_INCLUDED */
