//
// View.h
//
// $Id: //poco/Main/WebWidgets/include/Poco/WebWidgets/View.h#5 $
//
// Library: WebWidgets
// Package: Core
// Module:  View
//
// Definition of the View class.
//
// Copyright (c) 2008, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef WebWidgets_View_INCLUDED
#define WebWidgets_View_INCLUDED


#include "Poco/WebWidgets/Renderable.h"


namespace Poco {
namespace WebWidgets {


class WebWidgets_API View: public Renderable
	/// View is the base class for all visible objects; it defines basic 
	/// rendering and event architecture. In addition to the system-generated 
	/// numerical ID that every Renderable object has, a View has a user-defined 
	/// name, which can be used to find a view within a user interface.
	///
	/// Within a View hierarchy, a specific View can be found using its View 
	/// Path - a concatenation of the names of all parent views and the View's 
	/// name, separated by forward slashes.
{
public:
	typedef Poco::AutoPtr<View> Ptr;
	
	struct WebWidgets_API Pos
	{
		int posX;
		int posY;
		
		Pos();
			/// Creates a default position (posX = posY = -1)
			
		Pos(int posX, int posY);
	};
	
	const std::string& getName() const;
		/// Returns the View's name.
		
	void setName(const std::string& name);
		/// Sets the View's name.
		
	std::string path() const;
		/// Returns the View path for this View.
		
	View::Ptr find(const std::string& path) const;
		/// Finds a view by its view path.
		///
		/// A view path is a concatenation of View names, separated
		/// by a slash. A view path can be absolute (starts with a slash),
		/// or relative (starts with a name).
		///
		/// If an absolute path is given, it is searched starting from root().
		/// Otherwise, it is searched from the current view.
		///
		/// Returns a pointer to the found View, or a null pointer if
		/// the View could not be found.
		///
		/// Returns the current View if an empty path is given.
		
	virtual View::Ptr findChild(const std::string& name) const;
		/// Returns the View's child with the given name, or
		/// a null pointer if no child with the given name exists.
		///
		/// The default implementation returns null.
		///
		/// Must be overridded by each View class that can have
		/// children.
	
	View::Ptr parent() const;
		/// Returns the View's parent.
		
	View::Ptr root() const;
		/// Returns the top-most View in this View's hierarchy.
	
	void show(bool visible = true);
		/// Shows (or hides, if visible == false) the View.
		
	void hide();
		/// Hides the View by calling show(false).
		
	bool isVisible() const;
		/// Returns true iff the View is visible.
		
	virtual void setText(const std::string& text);
		/// Sets the text (title, caption, etc.) of this View.
		///
		/// The default implementation ignores the given text.
		
	virtual std::string getText() const;
		/// Returns the text of this View.
		///
		/// The default implementation always returns an empty string.
		
	virtual void setToolTip(const std::string& text);
		/// Sets the tooltip text for this View.
		///
		/// The default implementation ignores the given text.
		
	virtual std::string getToolTip() const;
		/// Returns the tooltip text for this View.
		///
		/// The default implementation always returns an empty string.
	
	void setWidth(Poco::UInt32 w);
		/// Sets the width of the view. 0 equals auto-size (default).

	void setHeight(Poco::UInt32 h);
		/// Sets the height of the view. 0 equals auto-size (default).

	Poco::UInt32 getWidth() const;
		/// Returns the width of the view

	Poco::UInt32 getHeight() const;
		/// Returns the height of the view
		
	bool hasPosition() const;
		/// Returns true if the view has a position not equal to the default one
		
	const View::Pos& getPosition() const;
		/// Returns the position
		
	void setPosition(int posX, int posY);
		/// Updates the position
	
	void setPosition(const View::Pos& pos);
		/// Updates the position
		
	void setClass(const std::string cls);
		/// Sets css class
		
	const std::string& getClass() const;
		/// Returns css class
		
	bool hasClass() const;
		/// Checks if a css value is set

	void showBorder(bool show);
		/// Show the border around the view
	
	bool showBorder() const;	
		/// Returns if view border is shown
	
protected:
	View(const std::string& name, const std::type_info& type);
		/// Creates a View and assigns it the given name.
		
	View(const std::type_info& type);
		/// Creates a View.
		
	~View();
		/// Destroys the View.
		
	void setParent(View* pParent);
		/// Sets the View's parent.
		
	void adoptChild(View* pChild);
		/// Sets the given View's parent to this View.

	void rejectChild(View* pChild);
		/// Sets the View's parent to null.
		
	virtual void onShow();
		/// Called when show(true) has been called. Can be overridden
		/// by subclasses to perform subclass-specific tasks.

	virtual void onHide();
		/// Called when show(false) has been called. Can be overridden
		/// by subclasses to perform subclass-specific tasks.
				
private:
	std::string  _name;
	View*        _pParent;
	bool         _visible;
	bool        _showBorder;
	Poco::UInt32 _width;
	Poco::UInt32 _height;
	Pos          _pos;
	std::string _class;

};


//
// inlines
//
inline const std::string& View::getName() const
{
	return _name;
}


inline View::Ptr View::parent() const
{
	return View::Ptr(_pParent, true);
}


inline void View::setParent(View* pParent)
{
	_pParent = pParent;
}

	
inline void View::adoptChild(View* pChild)
{
	if (pChild)
		pChild->setParent(this);
}


inline void View::rejectChild(View* pChild)
{
	if (pChild)
		pChild->setParent(0);
}


inline void View::hide()
{
	show(false);
}


inline bool View::isVisible() const
{
	return _visible;
}


inline void View::setWidth(Poco::UInt32 w)
{
	_width = w;
}


inline void View::setHeight(Poco::UInt32 h)
{
	_height = h;
}


inline Poco::UInt32 View::getWidth() const
{
	return _width;
}


inline Poco::UInt32 View::getHeight() const
{
	return _height;
}


inline bool View::hasPosition() const
{
	return !(_pos.posX == -1 && _pos.posY == -1);
}

	
inline const View::Pos& View::getPosition() const
{
	return _pos;
}

	
inline void View::setPosition(int posX, int posY)
{
	_pos.posX = posX;
	_pos.posY = posY;
}


inline void View::setPosition(const View::Pos& pos)
{
	_pos = pos;
}


inline void View::setClass(const std::string cls)
{
	_class = cls;
}

		
inline const std::string& View::getClass() const
{
	return _class;
}

		
inline bool View::hasClass() const
{
	return !getClass().empty();
}


inline void View::showBorder(bool show)
{
	_showBorder = show;
}


inline bool View::showBorder() const
{
	return _showBorder;
}

} } // namespace Poco::WebWidgets


#endif // WebWidgets_View_INCLUDED
