//
// ProgressIndicator.h
//
// $Id: //poco/Main/WebWidgets/include/Poco/WebWidgets/ProgressIndicator.h#2 $
//
// Library: WebWidgets
// Package: Views
// Module:  ProgressIndicator
//
// Definition of the ProgressIndicator class.
//
// Copyright (c) 2007, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef WebWidgets_ProgressIndicator_INCLUDED
#define WebWidgets_ProgressIndicator_INCLUDED


#include "Poco/WebWidgets/View.h"
#include "Poco/WebWidgets/Event.h"
#include "Poco/BasicEvent.h"


namespace Poco {
namespace WebWidgets {


class WebWidgets_API ProgressIndicator: public View
	/// class ProgressIndicator
{
public:
	typedef Poco::AutoPtr<ProgressIndicator> Ptr;
	typedef Event<ProgressIndicator> ProgressEvent;

	Poco::BasicEvent<ProgressEvent> progressChanged;
		/// Thrown after the progress value changed

public:
	enum UpdateMode
	{
		UM_AUTOMATIC = 0,
		UM_MANUAL
	};

	ProgressIndicator();
		/// Creates the ProgressIndicator with a default range of 0 and 100 and automatic mode

	ProgressIndicator(const std::string& name, const std::string& text, int min=0, int max=100, ProgressIndicator::UpdateMode m = UM_AUTOMATIC);
		/// Creates the ProgressIndicator. min und max define the range in which progress can be set(0-100 per default), the mode
		/// defines if we have a long running task where we cannot determine progress easily or if we use manual mode, ie. progress view
		/// is only updated when the user calls setProgress methods.

	int getMin() const;
		/// Returns the lower bound of the range

	int getMax() const;
		/// Returns the upper bound of the range

	int getProgress() const;
		/// Returns the current progress, guaranteed to be in range min, max

	void setProgress(int progress);
		/// Sets progress, if the value is too small or too large it will automatically be
		/// mapped to min/max

	double getProgressPercent() const;
		/// Returns the current progress in percent, is a value in range 0.0 to 1.0

	ProgressIndicator::UpdateMode getUpdateMode() const;
		/// Returns the current UpdateMode

	void setUpdateMode(ProgressIndicator::UpdateMode vm);
		/// Sets the UpdateMode

	std::string getText() const;
		/// Returns the text of the ProgressIndicator

	void setText(const std::string& text);
		/// Sets the text

protected:
	~ProgressIndicator();
		/// Destroys the ProgressIndicator.

private:
	std::string _text;
	int         _min;
	int         _max;
	int         _progress;
	UpdateMode  _updateMode;
};


//
// inlines
//

inline int ProgressIndicator::getMin() const
{
	return _min;
}


inline int ProgressIndicator::getMax() const
{
	return _max;
}


inline int ProgressIndicator::getProgress() const
{
	return _progress;
}


inline double ProgressIndicator::getProgressPercent() const
{
	return ((double)(_progress-_min))/(_max-_min);
}


inline ProgressIndicator::UpdateMode ProgressIndicator::getUpdateMode() const
{
	return _updateMode;
}


inline void ProgressIndicator::setUpdateMode(ProgressIndicator::UpdateMode um)
{
	_updateMode = um;
}


inline std::string ProgressIndicator::getText() const
{
	return _text;
}


inline void ProgressIndicator::setText(const std::string& text)
{
	_text = text;
}


} } // namespace Poco::WebWidgets


#endif // WebWidgets_ProgressIndicator_INCLUDED
