//
// AbstractConfigurationTest.cpp
//
// $Id: //poco/svn/Util/testsuite/src/AbstractConfigurationTest.cpp#1 $
//
// Copyright (c) 2004-2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#include "AbstractConfigurationTest.h"
#include "CppUnit/TestCaller.h"
#include "CppUnit/TestSuite.h"
#include "Poco/Util/MapConfiguration.h"
#include "Poco/AutoPtr.h"
#include "Poco/Exception.h"
#include <algorithm>


using Poco::Util::AbstractConfiguration;
using Poco::Util::MapConfiguration;
using Poco::AutoPtr;


AbstractConfigurationTest::AbstractConfigurationTest(const std::string& name): CppUnit::TestCase(name)
{
}


AbstractConfigurationTest::~AbstractConfigurationTest()
{
}


void AbstractConfigurationTest::testHasProperty()
{
	AutoPtr<AbstractConfiguration> pConf = createConfiguration();

	assert (pConf->hasProperty("prop1"));
	assert (pConf->hasProperty("prop2"));
	assert (pConf->hasProperty("prop3.string1"));
	assert (!pConf->hasProperty("prop3.string3"));
	assert (!pConf->hasProperty("foobar"));
}


void AbstractConfigurationTest::testGetString()
{
	AutoPtr<AbstractConfiguration> pConf = createConfiguration();
	
	assert (pConf->getString("prop1") == "foo");
	assert (pConf->getString("prop2") == "bar");
	assert (pConf->getString("prop3.string1") == "foo");
	assert (pConf->getString("prop3.string2") == "bar");
	assert (pConf->getString("ref1") == "foobar");
	assert (pConf->getRawString("ref1") == "${prop3.string1}${prop3.string2}");
	
	try
	{
		std::string res = pConf->getString("foo");
		fail("nonexistent property - must throw");
	}
	catch (Poco::NotFoundException&)
	{
	}
	
	assert (pConf->getString("prop1", "FOO") == "foo");
	assert (pConf->getString("prop2", "BAR") == "bar");
	assert (pConf->getString("prop3.string1", "FOO") == "foo");
	assert (pConf->getString("prop3.string2", "BAR") == "bar");
	assert (pConf->getString("prop3.string3", "FOOBAR") == "FOOBAR");
}


void AbstractConfigurationTest::testGetInt()
{
	AutoPtr<AbstractConfiguration> pConf = createConfiguration();

	assert (pConf->getInt("prop4.int1") == 42);
	assert (pConf->getInt("prop4.int2") == -42);
	assert (pConf->getInt("prop4.hex") == 0x1f);
	assert (pConf->getInt("ref2") == 42);
	
	try
	{
		int x = pConf->getInt("prop1");
		fail("not a number - must throw");
	}
	catch (Poco::SyntaxException&)
	{
	}
	
	assert (pConf->getInt("prop4.int1", 100) == 42);
	assert (pConf->getInt("prop4.int2", 100) == -42);
	assert (pConf->getInt("prop4.int3", 100) == 100);
}


void AbstractConfigurationTest::testGetDouble()
{
	AutoPtr<AbstractConfiguration> pConf = createConfiguration();

	assert (pConf->getDouble("prop4.double1") == 1);
	assert (pConf->getDouble("prop4.double2") == -1.5);
	
	try
	{
		double x = pConf->getDouble("prop1");
		fail("not a number - must throw");
	}
	catch (Poco::SyntaxException&)
	{
	}
	
	assert (pConf->getDouble("prop4.double1", 123.5) == 1);
	assert (pConf->getDouble("prop4.double2", 123.5) == -1.5);
	assert (pConf->getDouble("prop4.double3", 123.5) == 123.5);
}


void AbstractConfigurationTest::testGetBool()
{
	AutoPtr<AbstractConfiguration> pConf = createConfiguration();

	assert (pConf->getBool("prop4.bool1"));
	assert (!pConf->getBool("prop4.bool2"));
	assert (pConf->getBool("prop4.bool3"));
	assert (!pConf->getBool("prop4.bool4"));
	assert (pConf->getBool("prop4.bool5"));
	assert (!pConf->getBool("prop4.bool6"));
	assert (pConf->getBool("prop4.bool7"));
	assert (!pConf->getBool("prop4.bool8"));

	try
	{
		bool x = pConf->getBool("prop1");
		fail("not a boolean - must throw");
	}
	catch (Poco::SyntaxException&)
	{
	}

	assert (pConf->getBool("prop4.bool1", false));
	assert (!pConf->getBool("prop4.bool2", true));
	assert (pConf->getBool("prop4.boolx", true));
	assert (!pConf->getBool("prop4.booly", false));
}


void AbstractConfigurationTest::testExpand()
{
	AutoPtr<AbstractConfiguration> pConf = createConfiguration();

	assert (pConf->getString("ref1") == "foobar");
	assert (pConf->getInt("ref2") == 42);
	
	try
	{
		std::string s = pConf->getString("ref3");
		fail("circular reference - must throw");
	}
	catch (Poco::CircularReferenceException&)
	{
	}
	
	assert (pConf->getString("ref5") == "${refx}");
	assert (pConf->getString("ref6") == "${refx}");
	
	assert (pConf->expand("answer=${prop4.int1}") == "answer=42");
	assert (pConf->expand("bool5='${prop4.bool5}'") == "bool5='Yes'");
	assert (pConf->expand("undef='${undef}'") == "undef='${undef}'");
	assert (pConf->expand("deep='${ref1}'") == "deep='foobar'");
	assert (pConf->expand("deep='${ref7}'") == "deep='foobar'");
	
	assert (pConf->getString("dollar.atend") == "foo$");
	assert (pConf->getString("dollar.middle") == "foo$bar");
}


void AbstractConfigurationTest::testSetString()
{
	AutoPtr<AbstractConfiguration> pConf = createConfiguration();

	pConf->setString("set.string1", "foobar");
	pConf->setString("set.string2", "");
	assert (pConf->getString("set.string1") == "foobar");
	assert (pConf->getString("set.string2") == "");
}

void AbstractConfigurationTest::testSetInt()
{
	AutoPtr<AbstractConfiguration> pConf = createConfiguration();

	pConf->setInt("set.int1", 42);
	pConf->setInt("set.int2", -100);
	assert (pConf->getInt("set.int1") == 42);
	assert (pConf->getInt("set.int2") == -100);
}


void AbstractConfigurationTest::testSetDouble()
{
	AutoPtr<AbstractConfiguration> pConf = createConfiguration();

	pConf->setDouble("set.double1", 1.5);
	pConf->setDouble("set.double2", -1.5);
	assert (pConf->getDouble("set.double1") == 1.5);
	assert (pConf->getDouble("set.double2") == -1.5);	
}


void AbstractConfigurationTest::testSetBool()
{
	AutoPtr<AbstractConfiguration> pConf = createConfiguration();

	pConf->setBool("set.bool1", true);
	pConf->setBool("set.bool2", false);
	assert (pConf->getBool("set.bool1"));
	assert (!pConf->getBool("set.bool2"));
}	


void AbstractConfigurationTest::testKeys()
{
	AutoPtr<AbstractConfiguration> pConf = createConfiguration();

	AbstractConfiguration::Keys keys;
	pConf->keys(keys);
	assert (keys.size() == 12);
	assert (std::find(keys.begin(), keys.end(), "prop1") != keys.end());
	assert (std::find(keys.begin(), keys.end(), "prop2") != keys.end());
	assert (std::find(keys.begin(), keys.end(), "prop3") != keys.end());
	assert (std::find(keys.begin(), keys.end(), "prop4") != keys.end());
	assert (std::find(keys.begin(), keys.end(), "ref1") != keys.end());
	assert (std::find(keys.begin(), keys.end(), "ref2") != keys.end());
	assert (std::find(keys.begin(), keys.end(), "ref3") != keys.end());
	assert (std::find(keys.begin(), keys.end(), "ref4") != keys.end());
	assert (std::find(keys.begin(), keys.end(), "ref5") != keys.end());
	assert (std::find(keys.begin(), keys.end(), "ref6") != keys.end());
	assert (std::find(keys.begin(), keys.end(), "ref7") != keys.end());
	assert (std::find(keys.begin(), keys.end(), "dollar") != keys.end());

	pConf->keys("prop1", keys);
	assert (keys.empty());
	
	pConf->keys("prop3", keys);
	assert (keys.size() == 2);
	assert (std::find(keys.begin(), keys.end(), "string1") != keys.end());
	assert (std::find(keys.begin(), keys.end(), "string2") != keys.end());
}


AbstractConfiguration* AbstractConfigurationTest::createConfiguration() const
{
	AbstractConfiguration* pConfig = new MapConfiguration;
	
	pConfig->setString("prop1", "foo");
	pConfig->setString("prop2", "bar");
	pConfig->setString("prop3.string1", "foo");
	pConfig->setString("prop3.string2", "bar");
	pConfig->setString("prop4.int1", "42");
	pConfig->setString("prop4.int2", "-42");
	pConfig->setString("prop4.hex", "0x1f");
	pConfig->setString("prop4.double1", "1");
	pConfig->setString("prop4.double2", "-1.5");
	pConfig->setString("prop4.bool1", "1");
	pConfig->setString("prop4.bool2", "0");
	pConfig->setString("prop4.bool3", "True");
	pConfig->setString("prop4.bool4", "FALSE");
	pConfig->setString("prop4.bool5", "Yes");
	pConfig->setString("prop4.bool6", "no");
	pConfig->setString("prop4.bool7", "ON");
	pConfig->setString("prop4.bool8", "Off");
	pConfig->setString("ref1", "${prop3.string1}${prop3.string2}");
	pConfig->setString("ref2", "${prop4.int1}");
	pConfig->setString("ref3", "${ref4}");
	pConfig->setString("ref4", "${ref3}");
	pConfig->setString("ref5", "${refx}");
	pConfig->setString("ref6", "${refx");
	pConfig->setString("ref7", "${ref1}");
	pConfig->setString("dollar.atend", "foo$");
	pConfig->setString("dollar.middle", "foo$bar");
	
	return pConfig;
}


void AbstractConfigurationTest::setUp()
{
}


void AbstractConfigurationTest::tearDown()
{
}


CppUnit::Test* AbstractConfigurationTest::suite()
{
	CppUnit::TestSuite* pSuite = new CppUnit::TestSuite("AbstractConfigurationTest");

	CppUnit_addTest(pSuite, AbstractConfigurationTest, testHasProperty);
	CppUnit_addTest(pSuite, AbstractConfigurationTest, testGetString);
	CppUnit_addTest(pSuite, AbstractConfigurationTest, testGetInt);
	CppUnit_addTest(pSuite, AbstractConfigurationTest, testGetDouble);
	CppUnit_addTest(pSuite, AbstractConfigurationTest, testGetBool);
	CppUnit_addTest(pSuite, AbstractConfigurationTest, testExpand);
	CppUnit_addTest(pSuite, AbstractConfigurationTest, testSetString);
	CppUnit_addTest(pSuite, AbstractConfigurationTest, testSetInt);
	CppUnit_addTest(pSuite, AbstractConfigurationTest, testSetDouble);
	CppUnit_addTest(pSuite, AbstractConfigurationTest, testSetBool);
	CppUnit_addTest(pSuite, AbstractConfigurationTest, testKeys);

	return pSuite;
}
