//
// SMTPClientSession.cpp
//
// $Id: //poco/Main/Net/src/SMTPClientSession.cpp#10 $
//
// Library: Net
// Package: Mail
// Module:  SMTPClientSession
//
// Copyright (c) 2005-2008, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#include "Poco/Net/SMTPClientSession.h"
#include "Poco/Net/MailMessage.h"
#include "Poco/Net/MailRecipient.h"
#include "Poco/Net/MailStream.h"
#include "Poco/Net/SocketAddress.h"
#include "Poco/Net/SocketStream.h"
#include "Poco/Net/NetException.h"
#include "Poco/Environment.h"
#include "Poco/Net/NetworkInterface.h"
#include "Poco/HMACEngine.h"
#include "Poco/MD5Engine.h"
#include "Poco/DigestStream.h"
#include "Poco/StreamCopier.h"
#include "Poco/Base64Encoder.h"
#include "Poco/Base64Decoder.h"
#include <sstream>
#include <fstream>
#include <iostream>


using Poco::DigestEngine;
using Poco::HMACEngine;
using Poco::MD5Engine;
using Poco::DigestOutputStream;
using Poco::StreamCopier;
using Poco::Base64Encoder;
using Poco::Base64Decoder;
using Poco::Environment;


namespace Poco {
namespace Net {


SMTPClientSession::SMTPClientSession(const StreamSocket& socket):
	_socket(socket),
	_isOpen(false)
{
}


SMTPClientSession::SMTPClientSession(const std::string& host, Poco::UInt16 port):
	_socket(SocketAddress(host, port)),
	_isOpen(false)
{
}


SMTPClientSession::~SMTPClientSession()
{
	try
	{
		close();
	}
	catch (...)
	{
	}
}


void SMTPClientSession::setTimeout(const Poco::Timespan& timeout)
{
	_socket.setReceiveTimeout(timeout);
}

	
Poco::Timespan SMTPClientSession::getTimeout() const
{
	return _socket.getReceiveTimeout();
}


void SMTPClientSession::login(const std::string& hostname, std::string& response)
{
	open();
	int status = sendCommand("EHLO", hostname, response);
	if (isPermanentNegative(status))
		status = sendCommand("HELO", hostname, response);
	if (!isPositiveCompletion(status)) throw SMTPException("Login failed", response);
}


void SMTPClientSession::login(const std::string& hostname)
{
	std::string response;
	login(hostname, response);
}


void SMTPClientSession::login()
{
	login(Environment::nodeName());
}


void SMTPClientSession::loginUsingCRAM_MD5(const std::string& username, const std::string& password)
{
	int status = 0;
	std::string response;
	
	status = sendCommand("AUTH CRAM-MD5", response);
	if (!isPositiveIntermediate(status)) throw SMTPException("Cannot authenticate CRAM-MD5", response);
	std::string challengeBase64 = response.substr(4);
	
	std::istringstream istr(challengeBase64);
	Base64Decoder decoder(istr);
	std::string challenge;
	decoder >> challenge;
	
	HMACEngine<MD5Engine> hmac(password);
	hmac.update(challenge);
	
	const DigestEngine::Digest& digest = hmac.digest();
	std::string digestString(DigestEngine::digestToHex(digest));
	
	std::string challengeResponse = username + " " + digestString;
	
	std::ostringstream challengeResponseBase64;
	Base64Encoder encoder(challengeResponseBase64);
	encoder << challengeResponse;
	encoder.close();
	
	status = sendCommand(challengeResponseBase64.str(), response);
  	if (!isPositiveCompletion(status)) throw SMTPException("Login using CRAM-MD5 failed", response);  
}


void SMTPClientSession::loginUsingLogin(const std::string& username, const std::string& password)
{
	int status = 0;
	std::string response;
	
	status = sendCommand("AUTH LOGIN", response);
	if (!isPositiveIntermediate(status)) throw SMTPException("Cannot authenticate LOGIN", response);
	
	std::ostringstream usernameBase64;
	Base64Encoder usernameEncoder(usernameBase64);
	usernameEncoder << username;
	usernameEncoder.close();
	
	std::ostringstream passwordBase64;
	Base64Encoder passwordEncoder(passwordBase64);
	passwordEncoder << password;
	passwordEncoder.close();
	
	//Server request for username/password not defined could be either
	//S: login:
	//C: user_login
	//S: password:
	//C: user_password
	//or
	//S: password:
	//C: user_password
	//S: login:
	//C: user_login
	if (response == "334 VXNlcm5hbWU6") // username first (md5("Username:"))
	{
		status = sendCommand(usernameBase64.str(), response);
		if (!isPositiveIntermediate(status)) throw SMTPException("Login using LOGIN user name failed", response);
		
		status = sendCommand(passwordBase64.str(), response);
		if (!isPositiveCompletion(status)) throw SMTPException("Login using LOGIN password failed", response);  
	}
	else if (response == "334 UGFzc3dvcmQ6") // password first (md5("Password:"))
	{
		status = sendCommand(passwordBase64.str(), response);
		if (!isPositiveIntermediate(status)) throw SMTPException("Login using LOGIN password failed", response);  
		
		status = sendCommand(usernameBase64.str(), response);
		if (!isPositiveCompletion(status)) throw SMTPException("Login using LOGIN user name failed", response);
	}
  
}


void SMTPClientSession::login(LoginMethod loginMethod, const std::string& username, const std::string& password)
{
	std::string response;
	login(Environment::nodeName(), response);
	
	if (loginMethod == AUTH_CRAM_MD5)
	{
		if (response.find("CRAM-MD5", 0) != std::string::npos)
		{
			loginUsingCRAM_MD5(username, password);
		}
		else throw SMTPException("The mail service does not support CRAM-MD5 authentication", response);
	}
	else if (loginMethod == AUTH_LOGIN)
	{
		if (response.find("LOGIN", 0) != std::string::npos)
		{
			loginUsingLogin(username, password);
		}
		else throw SMTPException("The mail service does not support LOGIN authentication", response);
	}
	else if (loginMethod != AUTH_NONE)
	{
		throw SMTPException("The autentication method is not supported");
	}
}


void SMTPClientSession::open()
{
	if (!_isOpen)
	{
		std::string response;
		int status = _socket.receiveStatusMessage(response);
		if (!isPositiveCompletion(status)) throw SMTPException("The mail service is unavailable", response);
		_isOpen = true;
	}
}


void SMTPClientSession::close()
{
	if (_isOpen)
	{
		std::string response;
		sendCommand("QUIT", response);
		_socket.close();
		_isOpen = false;
	}
}


void SMTPClientSession::sendMessage(const MailMessage& message)
{
	std::string response;
	int status = 0;
	const std::string& fromField = message.getSender();
	std::string::size_type emailPos = fromField.find('<');
	if (emailPos == std::string::npos)
	{
		std::string sender("<");
		sender.append(fromField);
		sender.append(">");
		status = sendCommand("MAIL FROM:", sender, response);
	}
	else
	{
		status = sendCommand("MAIL FROM:", fromField.substr(emailPos, fromField.size() - emailPos), response);
	}
	if (!isPositiveCompletion(status)) throw SMTPException("Cannot send message", response);
	for (MailMessage::Recipients::const_iterator it = message.recipients().begin(); it != message.recipients().end(); ++it)
	{
		std::string recipient("<");
		recipient.append(it->getAddress());
		recipient.append(">");
		int status = sendCommand("RCPT TO:", recipient, response);
		if (!isPositiveCompletion(status)) throw SMTPException(std::string("Recipient rejected: ") + recipient, response);
	}
	status = sendCommand("DATA", response);
	if (!isPositiveIntermediate(status)) throw SMTPException("Cannot send message data", response);
	SocketOutputStream socketStream(_socket);
	MailOutputStream mailStream(socketStream);
	message.write(mailStream);
	mailStream.close();
	socketStream.flush();
	status = _socket.receiveStatusMessage(response);
	if (!isPositiveCompletion(status)) throw SMTPException("The server rejected the message", response);
}


int SMTPClientSession::sendCommand(const std::string& command, std::string& response)
{
	_socket.sendMessage(command);
	return _socket.receiveStatusMessage(response);
}


int SMTPClientSession::sendCommand(const std::string& command, const std::string& arg, std::string& response)
{
	_socket.sendMessage(command, arg);
	return _socket.receiveStatusMessage(response);
}


} } // namespace Poco::Net
