//
// TextConverterTest.cpp
//
// $Id: //poco/svn/Foundation/testsuite/src/TextConverterTest.cpp#2 $
//
// Copyright (c) 2004-2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#include "TextConverterTest.h"
#include "CppUnit/TestCaller.h"
#include "CppUnit/TestSuite.h"
#include "Poco/TextConverter.h"
#include "Poco/ASCIIEncoding.h"
#include "Poco/Latin1Encoding.h"
#include "Poco/UTF8Encoding.h"


using Poco::TextConverter;
using Poco::Latin1Encoding;
using Poco::UTF8Encoding;
using Poco::ASCIIEncoding;


TextConverterTest::TextConverterTest(const std::string& name): CppUnit::TestCase(name)
{
}


TextConverterTest::~TextConverterTest()
{
}


void TextConverterTest::testIdentityASCII()
{
	ASCIIEncoding encoding;
	TextConverter converter(encoding, encoding);
	
	std::string empty;
	std::string result0;
	int errors = converter.convert(empty, result0);
	assert (result0 == empty);
	assert (errors == 0);
	
	std::string fooBar = "foo bar";
	std::string result1;
	errors = converter.convert(fooBar, result1);
	assert (result1 == fooBar);
	assert (errors == 0);

	std::string result2;
	errors = converter.convert(fooBar.data(), (int) fooBar.length(), result2);
	assert (result2 == fooBar);
	assert (errors == 0);

	std::string result3;
	errors = converter.convert("", 0, result3);
	assert (result3.empty());
	assert (errors == 0);
	
	std::string x = "x";
	std::string result4;
	errors = converter.convert(x, result4);
	assert (result4 == x);
	assert (errors == 0);

	std::string result5;
	errors = converter.convert("x", 1, result5);
	assert (result5 == x);
	assert (errors == 0);
}


void TextConverterTest::testIdentityUTF8()
{
	UTF8Encoding encoding;
	TextConverter converter(encoding, encoding);
	
	std::string empty;
	std::string result0;
	int errors = converter.convert(empty, result0);
	assert (result0 == empty);
	assert (errors == 0);
	
	std::string fooBar = "foo bar";
	std::string result1;
	errors = converter.convert(fooBar, result1);
	assert (result1 == fooBar);
	assert (errors == 0);

	std::string result2;
	errors = converter.convert(fooBar.data(), (int) fooBar.length(), result2);
	assert (result2 == fooBar);
	assert (errors == 0);

	std::string result3;
	errors = converter.convert("", 0, result3);
	assert (result3.empty());
	assert (errors == 0);
	
	const unsigned char greek[] = {0x20, 0xce, 0xba, 0xe1, 0xbd, 0xb9, 0xcf, 0x83, 0xce, 0xbc, 0xce, 0xb5, 0x20, 0x00};
	std::string text((const char*) greek);
	
	std::string result4;
	errors = converter.convert(text, result4);
	assert (result4 == text);
	assert (errors == 0);
	
	std::string result5;
	errors = converter.convert((char*) greek, 13, result5);
	assert (result5 == text);
	assert (errors == 0);
	
	std::string x = "x";
	std::string result6;
	errors = converter.convert(x, result6);
	assert (result6 == x);
	assert (errors == 0);

	std::string result7;
	errors = converter.convert("x", 1, result7);
	assert (result7 == x);
	assert (errors == 0);
	
	std::string utfChar((char*) greek + 1, 2);
	std::string result8;
	errors = converter.convert(utfChar, result8);
	assert (result8 == utfChar);
	assert (errors == 0);
	
	std::string result9;
	errors = converter.convert((char*) greek + 1, 2, result9);
	assert (result9 == utfChar);
	assert (errors == 0);
}


void TextConverterTest::testUTF8toASCII()
{
	UTF8Encoding utf8Encoding;
	ASCIIEncoding asciiEncoding;
	TextConverter converter(utf8Encoding, asciiEncoding);

	const unsigned char greek[] = {0x20, 0xce, 0xba, 0xe1, 0xbd, 0xb9, 0xcf, 0x83, 0xce, 0xbc, 0xce, 0xb5, 0x20, 0x41, 0x42, 0x00};
	std::string text((const char*) greek);
	std::string result0;
	int errors = converter.convert(text, result0);
	assert (result0 == " ????? AB");
	assert (errors == 0);
	
	std::string result1;
	errors = converter.convert("abcde", 5, result1);
	assert (result1 == "abcde");
}


void TextConverterTest::testLatin1toUTF8()
{
	Latin1Encoding latin1Encoding;
	UTF8Encoding utf8Encoding;
	TextConverter converter(latin1Encoding, utf8Encoding);
	
	const unsigned char latin1Chars[] = {'g', 252, 'n', 't', 'e', 'r', 0};
	const unsigned char utf8Chars[]   = {'g', 195, 188, 'n', 't', 'e', 'r', 0};
	std::string latin1Text((const char*) latin1Chars);
	std::string utf8Text((const char*) utf8Chars);
	
	std::string result0;
	int errors = converter.convert(latin1Text, result0);
	assert (result0 == utf8Text);
	assert (errors == 0);
	
	std::string result1;
	errors = converter.convert(latin1Chars, 6, result1);
	assert (result0 == utf8Text);
	assert (errors == 0);
}


void TextConverterTest::testErrors()
{
	UTF8Encoding utf8Encoding;
	Latin1Encoding latin1Encoding;
	TextConverter converter(utf8Encoding, latin1Encoding);

	const unsigned char badChars[] = {'a', 'b', 255, 'c', 254, 0};
	std::string badText((const char*) badChars);
	
	std::string result;
	int errors = converter.convert(badText, result);
	assert (errors == 2);
}


void TextConverterTest::setUp()
{
}


void TextConverterTest::tearDown()
{
}


CppUnit::Test* TextConverterTest::suite()
{
	CppUnit::TestSuite* pSuite = new CppUnit::TestSuite("TextConverterTest");

	CppUnit_addTest(pSuite, TextConverterTest, testIdentityASCII);
	CppUnit_addTest(pSuite, TextConverterTest, testIdentityUTF8);
	CppUnit_addTest(pSuite, TextConverterTest, testUTF8toASCII);
	CppUnit_addTest(pSuite, TextConverterTest, testLatin1toUTF8);
	CppUnit_addTest(pSuite, TextConverterTest, testErrors);

	return pSuite;
}
