//
// RecordSet.cpp
//
// $Id: //poco/Main/Data/samples/RecordSet/src/RowFormatter.cpp#2 $
//
// This sample demonstrates the Data library recordset formatting
// capabilities.
//
// Copyright (c) 2008, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.


#include "Poco/SharedPtr.h"
#include "Poco/Data/SessionFactory.h"
#include "Poco/Data/Session.h"
#include "Poco/Data/Statement.h"
#include "Poco/Data/RecordSet.h"
#include "Poco/Data/RowFormatter.h"
#include "Poco/Data/SQLite/Connector.h"
#include <iostream>


using namespace Poco::Data::Keywords;
using Poco::Data::Session;
using Poco::Data::Statement;
using Poco::Data::RecordSet;
using Poco::Data::RowFormatter;


class HTMLTableFormatter : public RowFormatter
{
public:
	HTMLTableFormatter()
	{
		std::ostringstream os;
		os << "<TABLE border=\"1\" cellspacing=\"0\">" << std::endl;
		setPrefix(os.str());
		
		os.str("");
		os << "</TABLE>" << std::endl;
		setPostfix(os.str());
	}

	std::string& formatNames(const NameVecPtr pNames, std::string& formattedNames) const
	{
		std::ostringstream str;

		str << "\t<TR>" << std::endl;
		NameVec::const_iterator it = pNames->begin();
		NameVec::const_iterator end = pNames->end();
		for (; it != end; ++it)	str << "\t\t<TH align=\"center\">" << *it << "</TH>" << std::endl;
		str << "\t</TR>" << std::endl;

		return formattedNames = str.str();
	}

	std::string& formatValues(const ValueVec& vals, std::string& formattedValues) const
	{
		std::ostringstream str;

		str << "\t<TR>" << std::endl;
		ValueVec::const_iterator it = vals.begin();
		ValueVec::const_iterator end = vals.end();
		for (; it != end; ++it)
		{
			if (it->isNumeric()) 
				str << "\t\t<TD align=\"right\">";
			else 
				str << "\t\t<TD align=\"left\">";

			str << it->convert<std::string>() << "</TD>" << std::endl;
		}
		str << "\t</TR>" << std::endl;

		return formattedValues = str.str();
	}
};


int main(int argc, char** argv)
{
	// register SQLite connector
	Poco::Data::SQLite::Connector::registerConnector();
	
	// create a session
	Session session("SQLite", "sample.db");

	// drop sample table, if it exists
	session << "DROP TABLE IF EXISTS Simpsons", now;
	
	// (re)create table
	session << "CREATE TABLE Simpsons (Name VARCHAR(30), Address VARCHAR, Age INTEGER(3))", now;
	
	// insert some rows
	session << "INSERT INTO Simpsons VALUES('Homer Simpson', 'Springfield', 42)", now;
	session << "INSERT INTO Simpsons VALUES('Marge Simpson', 'Springfield', 38)", now;
	session << "INSERT INTO Simpsons VALUES('Bart Simpson', 'Springfield', 12)", now;
	session << "INSERT INTO Simpsons VALUES('Lisa Simpson', 'Springfield', 10)", now;
		
	// create a statement and print the column names and data as HTML table
	HTMLTableFormatter tf;
	Statement stmt = (session << "SELECT * FROM Simpsons", format(tf), now);
	RecordSet rs(stmt);
	std::cout << rs << std::endl;

	// Note: The code above is divided into individual steps for clarity purpose.
	// The four lines can be reduced to the following single line of code:
	std::cout << RecordSet(session, "SELECT * FROM Simpsons", new HTMLTableFormatter);

	// simple formatting example (uses the default SimpleRowFormatter provided by framework)
	std::cout << std::endl << "Simple formatting:" << std::endl << std::endl;
	std::cout << RecordSet(session, "SELECT * FROM Simpsons");

	return 0;
}
