//
// Time.h
//
// $Id: //poco/Main/Data/include/Poco/Data/Time.h#7 $
//
// Library: Data
// Package: DataCore
// Module:  Time
//
// Definition of the Time class.
//
// Copyright (c) 2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Data_Time_INCLUDED
#define Data_Time_INCLUDED


#include "Poco/Data/Data.h"
#include "Poco/Dynamic/VarHolder.h"
#include "Poco/Exception.h"


namespace Poco {

class DateTime;

namespace Data {


class Date;


class Data_API Time
	/// Time class wraps a DateTime and exposes time related interface.
	/// The purpose of this class is binding/extraction support for time fields.
{
public:
	Time();
		/// Creates the Time

	Time(int hour, int minute, int second);
		/// Creates the Time

	Time(const DateTime& dt);
		/// Creates the Time from DateTime

	~Time();
		/// Destroys the Time.

	int hour() const;
		/// Returns the hour.

	int minute() const;
		/// Returns the minute.

	int second() const;
		/// Returns the second.

	void assign(int hour, int minute, int second);
		/// Assigns time.

	Time& operator = (const DateTime& dt);
		/// Assignment operator for DateTime.

	bool operator == (const Time& time);
		/// Equality operator.

	bool operator != (const Time& time);
		/// Inequality operator.

	bool operator < (const Time& time);
		/// Less then operator.

	bool operator > (const Time& time);
		/// Greater then operator.

private:
	int _hour;
	int _minute;
	int _second;
};


//
// inlines
//
inline int Time::hour() const
{
	return _hour;
}


inline int Time::minute() const
{
	return _minute;
}


inline  int Time::second() const
{
	return _second;
}


inline Time& Time::operator = (const DateTime& dt)
{
	assign(dt.hour(), dt.minute(), dt.second());
	return *this;
}


inline bool Time::operator == (const Time& time)
{
	return _hour == time.hour() &&
		_minute == time.minute() &&
		_second == time.second();
}


inline bool Time::operator != (const Time& time)
{
	return !(*this == time);
}


inline bool Time::operator > (const Time& time)
{
	return !(*this == time) && !(*this < time);
}


} } // namespace Poco::Data


//
// VarHolderImpl<BLOB>
//


namespace Poco {
namespace Dynamic {


template <>
class VarHolderImpl<Poco::Data::Time>: public VarHolder
{
public:
	VarHolderImpl(const Poco::Data::Time& val): _val(val)
	{
	}

	~VarHolderImpl()
	{
	}
	
	const std::type_info& type() const
	{
		return typeid(Poco::Data::Time);
	}

	void convert(Poco::Timestamp& val) const
	{
		Poco::DateTime dt;
		dt.assign(dt.year(), dt.month(), dt.day(), _val.hour(), _val.minute(), _val.second());
		val = dt.timestamp();
	}

	void convert(Poco::DateTime& val) const
	{
		Poco::DateTime dt;
		dt.assign(dt.year(), dt.month(), dt.day(), _val.hour(), _val.minute(), _val.second());
		val = dt;
	}

	void convert(Poco::LocalDateTime& val) const
	{
		Poco::LocalDateTime ldt;
		ldt.assign(ldt.year(), ldt.month(), ldt.day(), _val.hour(), _val.minute(), _val.second());
		val = ldt;
	}

	void convert(std::string& val) const
	{
		DateTime dt(0, 1, 1, _val.hour(), _val.minute(), _val.second());
		val = DateTimeFormatter::format(dt, "%H:%M:%S");
	}

	VarHolder* clone() const
	{
		return new VarHolderImpl(_val);
	}
	
	const Poco::Data::Time& value() const
	{
		return _val;
	}

private:
	VarHolderImpl();
	Poco::Data::Time _val;
};


} } // namespace Poco::Dynamic


#endif // Data_Time_INCLUDED
