//
// RowFormatter.h
//
// $Id: //poco/Main/Data/include/Poco/Data/RowFormatter.h#1 $
//
// Library: Data
// Package: DataCore
// Module:  RowFormatter
//
// Definition of the RowFormatter class.
//
// Copyright (c) 2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Data_RowFormatter_INCLUDED
#define Data_RowFormatter_INCLUDED


#include "Poco/Data/Data.h"
#include "Poco/SharedPtr.h"
#include "Poco/RefCountedObject.h"
#include "Poco/Dynamic/Var.h"
#include <sstream>
#include <vector>


namespace Poco {
namespace Data {


class Data_API RowFormatter
	/// Row formatter is an abstract class providing definition for row formatting functionality.
	/// For custom formatting strategies, inherit from this class and override formatNames()
	/// and formatValues() member functions.
	///
	/// Row formatter can be either passed to the RecordSet at construction time,
	/// like in the following example:
	///
	/// RecordSet rs(session. "SELECT * FROM Table", new MyRowFormater);
	///
	/// or it can be supplied to the statement as in the following example:
	///
	/// MyRowFormatter rf
	/// session << "SELECT * FROM Table", format(rf);
	///
	/// If no formatter is externally supplied to the statement, the SimpleRowFormatter is used.
	/// Statement always has the ownership of the row formatter and shares
	/// it with rows through RecordSet.
	///
{
public:
	typedef std::vector<std::string>             NameVec;
	typedef SharedPtr<std::vector<std::string> > NameVecPtr;
	typedef std::vector<Poco::Dynamic::Var>            ValueVec;

	RowFormatter(const std::string& prefix = "", const std::string& postfix = "");
		/// Creates the RowFormatter and sets the prefix and postfix to specified values.

	virtual ~RowFormatter();
		/// Destroys the RowFormatter.

	virtual std::string& formatNames(const NameVecPtr pNames, std::string& formattedNames) const = 0;
		/// Formats the row field names.

	virtual std::string& formatValues(const ValueVec& vals, std::string& formattedValues) const = 0;
		/// Formats the row values.

	const std::string& prefix() const;
		/// Returns prefix string;

	const std::string& postfix() const;
		/// Returns postfix string;

protected:
	void setPrefix(const std::string& prefix);
	void setPostfix(const std::string& postfix);

private:
	std::string     _prefix;
	std::string     _postfix;
};


///
/// inlines
///


inline void RowFormatter::setPrefix(const std::string& prefix)
{
	_prefix = prefix;
}


inline void RowFormatter::setPostfix(const std::string& postfix)
{
	_postfix = postfix;
}


inline const std::string& RowFormatter::prefix() const
{
	return _prefix;
}


inline const std::string& RowFormatter::postfix() const
{
	return _postfix;
}


namespace Keywords {


template <typename T>
inline T* format(const T& formatter)
	/// Utility function used to pass formatter to the statement.
	/// Statement takes the ownership of the formatter.
{
	return new T(formatter);
}


} // namespace Keywords


typedef SharedPtr<RowFormatter> RowFormatterPtr;


} } // namespace Poco::Data


#endif // Data_RowFormatter_INCLUDED
